      SUBROUTINE XYS2000A ( DATE1, DATE2, X, Y, S )
*+
*  - - - - - - - - -
*   X Y S 2 0 0 0 A
*  - - - - - - - - -
*
*  Implementation of Capitaine et al. (2002) series for X, Y and s+XY/2,
*  consistent with the IAU 2000A precession-nutation model.
*
*  Annexe to IERS Conventions 2000, Chapter 5
*
*  Given:
*     DATE1,DATE2   d      TT date (JD = DATE1+DATE2)
*
*  Returned:
*     X             d      X-coordinate of CIP unit vector in GCRS
*     Y             d      Y-coordinate of CIP unit vector in GCRS
*     S             d      the quantity s, positioning the CEO
*
*  Calls SOFA routine iau_ANPM.
*
*  This revision:  2002 December 17
*
*-----------------------------------------------------------------------

      IMPLICIT NONE

      DOUBLE PRECISION DATE1, DATE2, X, Y, S

*  2Pi
      DOUBLE PRECISION D2PI
      PARAMETER ( D2PI = 6.283185307179586476925287D0 )

*  Arcseconds to radians
      DOUBLE PRECISION DAS2R
      PARAMETER ( DAS2R = 4.848136811095359935899141D-6 )

*  Reference epoch (J2000), JD
      DOUBLE PRECISION DJ0
      PARAMETER ( DJ0 = 2451545D0 )

*  Days per Julian century
      DOUBLE PRECISION DJC
      PARAMETER ( DJC = 36525D0 )

*  Time since J2000, in Julian centuries
      DOUBLE PRECISION T

*  Miscellaneous
      INTEGER I, J
      DOUBLE PRECISION A, S0, S1, S2, S3, S4, S5
      DOUBLE PRECISION iau_ANPM

*  Fundamental arguments
      DOUBLE PRECISION FA(14)

*  ------------------------------
*  The series for X: declarations
*  ------------------------------

*  Number of terms in the series
      INTEGER NXP, NX0, NX1, NX2, NX3, NX4
      PARAMETER ( NXP=6, NX0=1306, NX1=253, NX2=36, NX3=4, NX4=1 )

*  Polynomial coefficients
      DOUBLE PRECISION XP ( NXP )

*  Coefficients of l,l',F,D,Om,LMe,LVe,LE,LMa,LJu,LSa,LU,LN,pA
      INTEGER KX0 ( 14, NX0 ),
     :        KX1 ( 14, NX1 ),
     :        KX2 ( 14, NX2 ),
     :        KX3 ( 14, NX3 ),
     :        KX4 ( 14, NX4 )

*  Sine and cosine coefficients
      DOUBLE PRECISION SX0 ( 2, NX0 ),
     :                 SX1 ( 2, NX1 ),
     :                 SX2 ( 2, NX2 ),
     :                 SX3 ( 2, NX3 ),
     :                 SX4 ( 2, NX4 )

*  ------------------------------
*  The series for Y: declarations
*  ------------------------------

*  Number of terms in the series
      INTEGER NYP, NY0, NY1, NY2, NY3, NY4
      PARAMETER ( NYP=6, NY0= 962, NY1=277, NY2=30, NY3=5, NY4=1 )

*  Polynomial coefficients
      DOUBLE PRECISION YP ( NYP )

*  Coefficients of l,l',F,D,Om,LMe,LVe,LE,LMa,LJu,LSa,LU,LN,pA
      INTEGER KY0 ( 14, NY0 ),
     :        KY1 ( 14, NY1 ),
     :        KY2 ( 14, NY2 ),
     :        KY3 ( 14, NY3 ),
     :        KY4 ( 14, NY4 )

*  Sine and cosine coefficients
      DOUBLE PRECISION SY0 ( 2, NY0 ),
     :                 SY1 ( 2, NY1 ),
     :                 SY2 ( 2, NY2 ),
     :                 SY3 ( 2, NY3 ),
     :                 SY4 ( 2, NY4 )

*  -----------------------------------
*  The series for s+XY/2: declarations
*  -----------------------------------

*  Number of terms in the series
      INTEGER NSP, NS0, NS1, NS2, NS3, NS4
      PARAMETER ( NSP=6, NS0=  33, NS1=  3, NS2=25, NS3=4, NS4=1 )

*  Polynomial coefficients
      DOUBLE PRECISION SP ( NSP )

*  Coefficients of l,l',F,D,Om
      INTEGER KS0 ( 14, NS0 ),
     :        KS1 ( 14, NS1 ),
     :        KS2 ( 14, NS2 ),
     :        KS3 ( 14, NS3 ),
     :        KS4 ( 14, NS4 )

*  Sine and cosine coefficients
      DOUBLE PRECISION SS0 ( 2, NS0 ),
     :                 SS1 ( 2, NS1 ),
     :                 SS2 ( 2, NS2 ),
     :                 SS3 ( 2, NS3 ),
     :                 SS4 ( 2, NS4 )

*  ----------------------------------
*  The series for X: numerical values
*  ----------------------------------

*  Polynomial coefficients
      DATA XP /     -16616.99D-6,
     :          2004191742.88D-6,
     :             -427219.05D-6,
     :             -198620.54D-6,
     :                 -46.05D-6,
     :                   5.98D-6 /

*  Argument coefficients for t^0 terms
      DATA ( ( KX0(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   21,   30 ) /
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   31,   40 ) /
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   41,   50 ) /
     :  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   51,   60 ) /
     :  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1,
     :  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   61,   70 ) /
     :  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   71,   80 ) /
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2,
     :  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   81,   90 ) /
     :  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   91,  100 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
     :  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  101,  110 ) /
     :  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  111,  120 ) /
     :  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  121,  130 ) /
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  131,  140 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
     :  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
     :  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
     :  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  141,  150 ) /
     :  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  151,  160 ) /
     :  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  161,  170 ) /
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
     :  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  171,  180 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
     :  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  181,  190 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
     :  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  191,  200 ) /
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2,
     :  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2,
     :  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0,
     :  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  201,  210 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1,
     :  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2,
     :  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  211,  220 ) /
     :  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2,
     :  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  221,  230 ) /
     :  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2,
     :  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
     :  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  231,  240 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2,
     :  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
     :  1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  241,  250 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2,
     :  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
     :  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  251,  260 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
     :  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
     :  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1,
     :  1,  0,  0, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  261,  270 ) /
     :  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  271,  280 ) /
     :  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2,
     :  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  281,  290 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,
     :  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1,
     :  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  291,  300 ) /
     :  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0, -6,  8,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  301,  310 ) /
     :  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0,
     :  1,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  311,  320 ) /
     :  0,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2,
     :  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  6,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  321,  330 ) /
     :  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  331,  340 ) /
     :  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, -2,
     :  2,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0,
     :  3,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  341,  350 ) /
     :  2,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,
     :  0,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2,
     :  1, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  351,  360 ) /
     :  0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2,
     :  1,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  361,  370 ) /
     :  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0,
     :  3, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  371,  380 ) /
     :  0,  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,
     :  2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  381,  390 ) /
     :  1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,
     :  0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,
     :  3,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  391,  400 ) /
     :  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0, -1,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  1, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  401,  410 ) /
     :  3,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2,
     :  2,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,
     :  2, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  411,  420 ) /
     :  2,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -2,
     :  1,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  421,  430 ) /
     :  1,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0,
     :  2,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, -1,
     :  0,  0,  1, -1,  0,  0, -5,  7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  431,  440 ) /
     :  2,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -1,
     :  1, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  441,  450 ) /
     :  2, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2,
     :  2,  0,  0, -2,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,
     :  1, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  451,  460 ) /
     :  1, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0,
     :  2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,
     :  1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  461,  470 ) /
     :  2, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  471,  480 ) /
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0,
     :  1, -1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -2,
     :  1,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -1,
     :  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  481,  490 ) /
     :  0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0,
     :  1,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  491,  500 ) /
     :  0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,
     :  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  501,  510 ) /
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1,
     :  1,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  511,  520 ) /
     :  2,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,
     :  1,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  521,  530 ) /
     :  2,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2,
     :  1,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  531,  540 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0,
     :  0,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  541,  550 ) /
     :  3,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -2,
     :  1, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0, -2,  5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  551,  560 ) /
     :  3,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0, -1,  1,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -1,
     :  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  561,  570 ) /
     :  1,  0,  0, -1, -1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  5,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -9,  4,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  571,  580 ) /
     :  1,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0,
     :  0,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  581,  590 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,
     :  1, -1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  591,  600 ) /
     :  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0,
     :  2,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  601,  610 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,
     :  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0,
     :  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  611,  620 ) /
     :  2,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  621,  630 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,
     :  1,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2,
     :  0,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  631,  640 ) /
     :  3,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  641,  650 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2,
     :  1,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  651,  660 ) /
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6,-15,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2,
     :  1,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  661,  670 ) /
     :  0,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  4, -3,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  671,  680 ) /
     :  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0,
     :  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  3, -1,  0,  0,  0,
     :  2, -1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  681,  690 ) /
     :  0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  3,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0, -6,  8,  0,  0,  0,  0,  0,  0,
     :  1,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0,
     :  3,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  691,  700 ) /
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1, -5,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0,
     :  0,  0,  4, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0,
     :  0,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  0,  2,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  701,  710 ) /
     :  0,  0,  2, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  2,  6,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2,
     :  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  711,  720 ) /
     :  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0,
     :  1, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  721,  730 ) /
     :  3, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  731,  740 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1,
     :  1, -1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  0,  5,  0,  0,  0,
     :  3,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  741,  750 ) /
     :  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  1, -1, -1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0,
     :  3, -1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J =  751,  760 ) /
     :  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0, -2,
     :  1,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  761,  770 ) /
     :  2,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -5,  6,  0,  0,  0,  0,  0,
     :  0,  0,  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  3,  0,  0,  0,  0,
     :  1,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  771,  780 ) /
     :  0,  0,  4, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0,
     :  5,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9,-12,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  781,  790 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0,
     :  1,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  791,  800 ) /
     :  0,  2, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -1,  0,  0,  3, -7,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  2,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  801,  810 ) /
     :  1, -1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  811,  820 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  821,  830 ) /
     :  1,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0,
     :  0,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0,
     :  3, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -2,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  831,  840 ) /
     :  1,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0,
     :  3,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  841,  850 ) /
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  851,  860 ) /
     :  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  861,  870 ) /
     :  1, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  3,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0,
     :  1,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  871,  880 ) /
     :  3,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J =  881,  890 ) /
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1, -6,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  891,  900 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0,
     :  1, -1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0,
     :  0,  0,  2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  901,  910 ) /
     :  0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  1,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  2,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  911,  920 ) /
     :  2,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  4, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  921,  930 ) /
     :  1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  1, -1,  0,-18, 17,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  8,-16,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  931,  940 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2, -7,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,
     :  0,  1,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  941,  950 ) /
     :  2,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  951,  960 ) /
     :  1, -2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0,
     :  1,  0, -2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  961,  970 ) /
     :  0,  0,  2,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  971,  980 ) /
     :  4, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  4, -5,  0,  0,  0,
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  981,  990 ) /
     :  1,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0,
     :  1,  0, -1, -1, -1,  0, 20,-20,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  991, 1000 ) /
     :  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -5,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1001, 1010 ) /
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -5,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2, -2,
     :  0,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1011, 1020 ) /
     :  1,  0, -1,  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1021, 1030 ) /
     :  0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1, -1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  2,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1031, 1040 ) /
     :  1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  2,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1041, 1050 ) /
     :  3,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0, -4,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1051, 1060 ) /
     :  0,  0,  1, -1,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  0,  1,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  1, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  3,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  2,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1061, 1070 ) /
     :  2,  0,  0, -2,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0,
     :  2, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1071, 1080 ) /
     :  0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J = 1081, 1090 ) /
     :  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -8,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1091, 1100 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,
     :  2,  1,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1101, 1110 ) /
     :  3, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0, -4,  4,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1111, 1120 ) /
     :  3,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0,
     :  3,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1121, 1130 ) /
     :  0,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  0,  1,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1131, 1140 ) /
     :  1,  0,  2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -6,  8,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1141, 1150 ) /
     :  2,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  3, -1,  0,  0,  0,
     :  2, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1151, 1160 ) /
     :  5,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  2,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1161, 1170 ) /
     :  2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0, -1,  0,  2,  0,  0,  0,  0,  0,  0,
     :  4,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0, -1,  0,  0, -2,  0,  0,  0,
     :  1, -1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1171, 1180 ) /
     :  2,  0, -1, -1,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,
     :  4,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1181, 1190 ) /
     :  2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  5, -5,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  1,  5,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  0,  2,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0, -4,  4,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -2,  0,  0,  5, -9,  0,  0,  0,  0,  0,
     :  2,  0, -2, -5, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1191, 1200 ) /
     :  1,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  1,  0, -1,  0, -1,  0, -3,  5,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0,
     :  1,  0, -1, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  1,  0, -1,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1201, 1210 ) /
     :  0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1211, 1220 ) /
     :  0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  8,-14,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1221, 1230 ) /
     :  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  5, -6, -4,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1231, 1240 ) /
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  7,-12,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1241, 1250 ) /
     :  0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  5,-13,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5,-16,  4,  5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J = 1251, 1260 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -5,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -9,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -8,  1,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1261, 1270 ) /
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -6,  3,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1271, 1280 ) /
     :  1,  1,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  0,  5,  0,  0,  0,
     :  3,  0, -2, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1281, 1290 ) /
     :  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0,
     :  2,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1, -1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1291, 1300 ) /
     :  0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0,
     :  2,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1301,  NX0 ) /
     :  1,  0,  0, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  1,  0, -2,  4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^1 terms
      DATA ( ( KX1(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   21,   30 ) /
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   31,   40 ) /
     :  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   41,   50 ) /
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1,
     :  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   51,   60 ) /
     :  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2,
     :  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   61,   70 ) /
     :  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   71,   80 ) /
     :  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
     :  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   81,   90 ) /
     :  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1,
     :  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   91,  100 ) /
     :  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
     :  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2,
     :  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  101,  110 ) /
     :  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
     :  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  111,  120 ) /
     :  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2,
     :  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2 /
      DATA ( ( KX1(I,J), I=1,14), J =  121,  130 ) /
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  131,  140 ) /
     :  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
     :  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
     :  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
     :  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  141,  150 ) /
     :  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0,
     :  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1 /
      DATA ( ( KX1(I,J), I=1,14), J =  151,  160 ) /
     :  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2,
     :  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2,
     :  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  161,  170 ) /
     :  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
     :  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  171,  180 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0,
     :  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2,
     :  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2,
     :  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  181,  190 ) /
     :  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
     :  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2,
     :  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2,
     :  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  191,  200 ) /
     :  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1,
     :  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2,
     :  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2 /
      DATA ( ( KX1(I,J), I=1,14), J =  201,  210 ) /
     :  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
     :  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
     :  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
     :  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  211,  220 ) /
     :  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
     :  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  221,  230 ) /
     :  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1,
     :  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2,
     :  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2,
     :  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  231,  240 ) /
     :  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2,
     :  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2,
     :  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX1(I,J), I=1,14), J =  241,  250 ) /
     :  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0,
     :  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0,
     :  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  251,  NX1 ) /
     :  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^2 terms
      DATA ( ( KX2(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   21,   30 ) /
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   31,  NX2 ) /
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^3 terms
      DATA ( ( KX3(I,J), I=1,14), J =    1,  NX3 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^4 terms
      DATA ( ( KX4(I,J), I=1,14), J =    1,  NX4 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Sine and cosine coefficients for t^0 terms
      DATA ( ( SX0(I,J), I=1,2), J =    1,   10 ) /
     :         -6844318.44D-6,       +1328.67D-6,
     :          -523908.04D-6,        -544.76D-6,
     :           -90552.22D-6,        +111.23D-6,
     :           +82168.76D-6,         -27.64D-6,
     :           +58707.02D-6,        +470.05D-6,
     :           +28288.28D-6,         -34.69D-6,
     :           -20557.78D-6,         -20.84D-6,
     :           -15406.85D-6,         +15.12D-6,
     :           -11991.74D-6,         +32.46D-6,
     :            -8584.95D-6,          +4.42D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   11,   20 ) /
     :            -6245.02D-6,          -6.68D-6,
     :            +5095.50D-6,          +7.19D-6,
     :            -4910.93D-6,          +0.76D-6,
     :            +2521.07D-6,          -5.97D-6,
     :            +2511.85D-6,          +1.07D-6,
     :            +2372.58D-6,          +5.93D-6,
     :            +2307.58D-6,          -7.52D-6,
     :            -2053.16D-6,          +5.13D-6,
     :            +1898.27D-6,          -0.72D-6,
     :            -1825.49D-6,          +1.23D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   21,   30 ) /
     :            -1534.09D-6,          +6.29D-6,
     :            -1292.02D-6,          +0.00D-6,
     :            -1234.96D-6,          +5.21D-6,
     :            +1163.22D-6,          -2.94D-6,
     :            +1137.48D-6,          -0.04D-6,
     :            +1029.70D-6,          -2.63D-6,
     :             -866.48D-6,          +0.52D-6,
     :             -813.13D-6,          +0.40D-6,
     :             +664.57D-6,          -0.40D-6,
     :             -628.24D-6,          -0.64D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   31,   40 ) /
     :             -603.52D-6,          +0.44D-6,
     :             -556.26D-6,          +3.16D-6,
     :             -512.37D-6,          -1.47D-6,
     :             +506.65D-6,          +2.54D-6,
     :             +438.51D-6,          -0.56D-6,
     :             +405.91D-6,          +0.99D-6,
     :             -122.67D-6,        +203.78D-6,
     :             -305.78D-6,          +1.75D-6,
     :             +300.99D-6,          -0.44D-6,
     :             -292.37D-6,          -0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   41,   50 ) /
     :             +284.09D-6,          +0.32D-6,
     :             -264.02D-6,          +0.99D-6,
     :             +261.54D-6,          -0.95D-6,
     :             +256.30D-6,          -0.28D-6,
     :             -250.54D-6,          +0.08D-6,
     :             +230.72D-6,          +0.08D-6,
     :             +229.78D-6,          -0.60D-6,
     :             -212.82D-6,          +0.84D-6,
     :             +196.64D-6,          -0.84D-6,
     :             +188.95D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   51,   60 ) /
     :             +187.95D-6,          -0.24D-6,
     :             -160.15D-6,         -14.04D-6,
     :             -172.95D-6,          -0.40D-6,
     :             -168.26D-6,          +0.20D-6,
     :             +161.79D-6,          +0.24D-6,
     :             +161.34D-6,          +0.20D-6,
     :              +57.44D-6,         +95.82D-6,
     :             +142.16D-6,          +0.20D-6,
     :             -134.81D-6,          +0.20D-6,
     :             +132.81D-6,          -0.52D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   61,   70 ) /
     :             -130.31D-6,          +0.04D-6,
     :             +121.98D-6,          -0.08D-6,
     :             -115.40D-6,          +0.60D-6,
     :             -114.49D-6,          +0.32D-6,
     :             +112.14D-6,          +0.28D-6,
     :             +105.29D-6,          +0.44D-6,
     :              +98.69D-6,          -0.28D-6,
     :              +91.31D-6,          -0.40D-6,
     :              +86.74D-6,          -0.08D-6,
     :              -18.38D-6,         +63.80D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   71,   80 ) /
     :              +82.14D-6,          +0.00D-6,
     :              +79.03D-6,          -0.24D-6,
     :               +0.00D-6,         -79.08D-6,
     :              -78.56D-6,          +0.00D-6,
     :              +47.73D-6,         +23.79D-6,
     :              +66.03D-6,          -0.20D-6,
     :              +62.65D-6,          -0.24D-6,
     :              +60.50D-6,          +0.36D-6,
     :              +59.07D-6,          +0.00D-6,
     :              +57.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   81,   90 ) /
     :              -55.66D-6,          +0.16D-6,
     :              -54.81D-6,          -0.08D-6,
     :              -53.22D-6,          -0.20D-6,
     :              -52.95D-6,          +0.32D-6,
     :              -52.27D-6,          +0.00D-6,
     :              +51.32D-6,          +0.00D-6,
     :              -51.00D-6,          -0.12D-6,
     :              +51.02D-6,          +0.00D-6,
     :              -48.65D-6,          -1.15D-6,
     :              +48.29D-6,          +0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   91,  100 ) /
     :              -46.38D-6,          +0.00D-6,
     :              -45.59D-6,          -0.12D-6,
     :              -43.76D-6,          +0.36D-6,
     :              -40.58D-6,          -1.00D-6,
     :               +0.00D-6,         -41.53D-6,
     :              +40.54D-6,          -0.04D-6,
     :              +40.33D-6,          -0.04D-6,
     :              -38.57D-6,          +0.08D-6,
     :              +37.75D-6,          +0.04D-6,
     :              +37.15D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  101,  110 ) /
     :              +36.68D-6,          -0.04D-6,
     :              -18.30D-6,         -17.30D-6,
     :              -17.86D-6,         +17.10D-6,
     :              -34.81D-6,          +0.04D-6,
     :              -33.22D-6,          +0.08D-6,
     :              +32.43D-6,          -0.04D-6,
     :              -30.47D-6,          +0.04D-6,
     :              -29.53D-6,          +0.04D-6,
     :              +28.50D-6,          -0.08D-6,
     :              +28.35D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  111,  120 ) /
     :              -28.00D-6,          +0.00D-6,
     :              -27.61D-6,          +0.20D-6,
     :              -26.77D-6,          +0.08D-6,
     :              +26.54D-6,          -0.12D-6,
     :              +26.54D-6,          +0.04D-6,
     :              -26.17D-6,          +0.00D-6,
     :              -25.42D-6,          -0.08D-6,
     :              -16.91D-6,          +8.43D-6,
     :               +0.32D-6,         +24.42D-6,
     :              -19.53D-6,          +5.09D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  121,  130 ) /
     :              -23.79D-6,          +0.00D-6,
     :              +23.66D-6,          +0.00D-6,
     :              -23.47D-6,          +0.16D-6,
     :              +23.39D-6,          -0.12D-6,
     :              -23.49D-6,          +0.00D-6,
     :              -23.28D-6,          -0.08D-6,
     :              -22.99D-6,          +0.04D-6,
     :              -22.67D-6,          -0.08D-6,
     :               +9.35D-6,         +13.29D-6,
     :              +22.47D-6,          -0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  131,  140 ) /
     :               +4.89D-6,         -16.55D-6,
     :               +4.89D-6,         -16.51D-6,
     :              +21.28D-6,          -0.08D-6,
     :              +20.57D-6,          +0.64D-6,
     :              +21.01D-6,          +0.00D-6,
     :               +1.23D-6,         -19.13D-6,
     :              -19.97D-6,          +0.12D-6,
     :              +19.65D-6,          -0.08D-6,
     :              +19.58D-6,          -0.12D-6,
     :              +19.61D-6,          -0.08D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  141,  150 ) /
     :              -19.41D-6,          +0.08D-6,
     :              -19.49D-6,          +0.00D-6,
     :              -18.64D-6,          +0.00D-6,
     :              +18.58D-6,          +0.04D-6,
     :              -18.42D-6,          +0.00D-6,
     :              +18.22D-6,          +0.00D-6,
     :               -0.72D-6,         -17.34D-6,
     :              -18.02D-6,          -0.04D-6,
     :              +17.74D-6,          +0.08D-6,
     :              +17.46D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  151,  160 ) /
     :              -17.42D-6,          +0.00D-6,
     :               -6.60D-6,         +10.70D-6,
     :              +16.43D-6,          +0.52D-6,
     :              -16.75D-6,          +0.04D-6,
     :              +16.55D-6,          -0.08D-6,
     :              +16.39D-6,          -0.08D-6,
     :              +13.88D-6,          -2.47D-6,
     :              +15.69D-6,          +0.00D-6,
     :              -15.52D-6,          +0.00D-6,
     :               +3.34D-6,         +11.86D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  161,  170 ) /
     :              +14.72D-6,          -0.32D-6,
     :              +14.92D-6,          -0.04D-6,
     :               -3.26D-6,         +11.62D-6,
     :              -14.64D-6,          +0.00D-6,
     :               +0.00D-6,         +14.47D-6,
     :              -14.37D-6,          +0.00D-6,
     :              +14.32D-6,          -0.04D-6,
     :              -14.10D-6,          +0.04D-6,
     :              +10.86D-6,          +3.18D-6,
     :              -10.58D-6,          -3.10D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  171,  180 ) /
     :               -3.62D-6,          +9.86D-6,
     :              -13.48D-6,          +0.00D-6,
     :              +13.41D-6,          -0.04D-6,
     :              +13.32D-6,          -0.08D-6,
     :              -13.33D-6,          -0.04D-6,
     :              -13.29D-6,          +0.00D-6,
     :               -0.20D-6,         +13.05D-6,
     :               +0.00D-6,         +13.13D-6,
     :               -8.99D-6,          +4.02D-6,
     :              -12.93D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  181,  190 ) /
     :               +2.03D-6,         +10.82D-6,
     :              -12.78D-6,          +0.04D-6,
     :              +12.24D-6,          +0.04D-6,
     :               +8.71D-6,          +3.54D-6,
     :              +11.98D-6,          -0.04D-6,
     :              -11.38D-6,          +0.04D-6,
     :              -11.30D-6,          +0.00D-6,
     :              +11.14D-6,          -0.04D-6,
     :              +10.98D-6,          +0.00D-6,
     :              -10.98D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  191,  200 ) /
     :               +0.44D-6,         -10.38D-6,
     :              +10.46D-6,          +0.08D-6,
     :              -10.42D-6,          +0.00D-6,
     :              -10.30D-6,          +0.08D-6,
     :               +6.92D-6,          +3.34D-6,
     :              +10.07D-6,          +0.04D-6,
     :              +10.02D-6,          +0.00D-6,
     :               -9.75D-6,          +0.04D-6,
     :               +9.75D-6,          +0.00D-6,
     :               +9.67D-6,          -0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  201,  210 ) /
     :               -1.99D-6,          +7.72D-6,
     :               +0.40D-6,          +9.27D-6,
     :               -3.42D-6,          +6.09D-6,
     :               +0.56D-6,          -8.67D-6,
     :               -9.19D-6,          +0.00D-6,
     :               +9.11D-6,          +0.00D-6,
     :               +9.07D-6,          +0.00D-6,
     :               +1.63D-6,          +6.96D-6,
     :               -8.47D-6,          +0.00D-6,
     :               -8.28D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  211,  220 ) /
     :               +8.27D-6,          +0.04D-6,
     :               -8.04D-6,          +0.00D-6,
     :               +7.91D-6,          +0.00D-6,
     :               -7.84D-6,          -0.04D-6,
     :               -7.64D-6,          +0.08D-6,
     :               +5.21D-6,          -2.51D-6,
     :               -5.77D-6,          +1.87D-6,
     :               +5.01D-6,          -2.51D-6,
     :               -7.48D-6,          +0.00D-6,
     :               -7.32D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  221,  230 ) /
     :               +7.40D-6,          -0.04D-6,
     :               +7.44D-6,          +0.00D-6,
     :               +6.32D-6,          -1.11D-6,
     :               -6.13D-6,          -1.19D-6,
     :               +0.20D-6,          -6.88D-6,
     :               +6.92D-6,          +0.04D-6,
     :               +6.48D-6,          -0.48D-6,
     :               -6.94D-6,          +0.00D-6,
     :               +2.47D-6,          -4.46D-6,
     :               -2.23D-6,          -4.65D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  231,  240 ) /
     :               -1.07D-6,          -5.69D-6,
     :               +4.97D-6,          -1.71D-6,
     :               +5.57D-6,          +1.07D-6,
     :               -6.48D-6,          +0.08D-6,
     :               +2.03D-6,          +4.53D-6,
     :               +4.10D-6,          -2.39D-6,
     :               +0.00D-6,          -6.44D-6,
     :               -6.40D-6,          +0.00D-6,
     :               +6.32D-6,          +0.00D-6,
     :               +2.67D-6,          -3.62D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  241,  250 ) /
     :               -1.91D-6,          -4.38D-6,
     :               -2.43D-6,          -3.82D-6,
     :               +6.20D-6,          +0.00D-6,
     :               -3.38D-6,          -2.78D-6,
     :               -6.12D-6,          +0.04D-6,
     :               -6.09D-6,          -0.04D-6,
     :               -6.01D-6,          -0.04D-6,
     :               +3.18D-6,          -2.82D-6,
     :               -5.05D-6,          +0.84D-6,
     :               +5.85D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  251,  260 ) /
     :               +5.69D-6,          -0.12D-6,
     :               +5.73D-6,          -0.04D-6,
     :               +5.61D-6,          +0.00D-6,
     :               +5.49D-6,          +0.00D-6,
     :               -5.33D-6,          +0.04D-6,
     :               -5.29D-6,          +0.00D-6,
     :               +5.25D-6,          +0.00D-6,
     :               +0.99D-6,          +4.22D-6,
     :               -0.99D-6,          +4.22D-6,
     :               +0.00D-6,          +5.21D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  261,  270 ) /
     :               +5.13D-6,          +0.04D-6,
     :               -4.90D-6,          +0.00D-6,
     :               -3.10D-6,          +1.79D-6,
     :               -4.81D-6,          +0.04D-6,
     :               -4.75D-6,          +0.00D-6,
     :               +4.70D-6,          -0.04D-6,
     :               -4.69D-6,          +0.00D-6,
     :               -4.65D-6,          +0.00D-6,
     :               +4.65D-6,          +0.00D-6,
     :               -4.57D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  271,  280 ) /
     :               +4.49D-6,          -0.04D-6,
     :               -4.53D-6,          +0.00D-6,
     :               +0.00D-6,          -4.53D-6,
     :               +0.00D-6,          -4.53D-6,
     :               -4.53D-6,          +0.00D-6,
     :               +4.50D-6,          +0.00D-6,
     :               -4.49D-6,          +0.00D-6,
     :               +1.83D-6,          +2.63D-6,
     :               +4.38D-6,          +0.00D-6,
     :               +0.88D-6,          -3.46D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  281,  290 ) /
     :               -2.70D-6,          +1.55D-6,
     :               -4.22D-6,          +0.00D-6,
     :               -4.10D-6,          -0.12D-6,
     :               +3.54D-6,          -0.64D-6,
     :               -3.50D-6,          +0.68D-6,
     :               +4.18D-6,          +0.00D-6,
     :               +4.14D-6,          +0.00D-6,
     :               +4.10D-6,          +0.00D-6,
     :               -4.06D-6,          +0.00D-6,
     :               +2.70D-6,          -1.35D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  291,  300 ) /
     :               -4.04D-6,          +0.00D-6,
     :               -3.98D-6,          -0.04D-6,
     :               -3.98D-6,          +0.04D-6,
     :               +4.02D-6,          +0.00D-6,
     :               +3.94D-6,          +0.00D-6,
     :               +0.84D-6,          -3.10D-6,
     :               +3.30D-6,          +0.60D-6,
     :               -1.59D-6,          +2.27D-6,
     :               -3.66D-6,          -0.20D-6,
     :               -3.10D-6,          -0.72D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  301,  310 ) /
     :               -3.82D-6,          +0.00D-6,
     :               -3.62D-6,          -0.16D-6,
     :               -3.74D-6,          +0.00D-6,
     :               +3.74D-6,          +0.00D-6,
     :               -3.74D-6,          +0.00D-6,
     :               -3.71D-6,          +0.00D-6,
     :               +3.02D-6,          +0.68D-6,
     :               +3.70D-6,          +0.00D-6,
     :               +3.30D-6,          +0.40D-6,
     :               -3.66D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  311,  320 ) /
     :               +3.66D-6,          +0.04D-6,
     :               -3.62D-6,          +0.00D-6,
     :               -3.61D-6,          +0.00D-6,
     :               -2.90D-6,          +0.68D-6,
     :               +0.80D-6,          -2.78D-6,
     :               +3.54D-6,          +0.00D-6,
     :               -3.54D-6,          +0.00D-6,
     :               -3.50D-6,          +0.00D-6,
     :               +3.45D-6,          +0.00D-6,
     :               +0.00D-6,          -3.42D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  321,  330 ) /
     :               +3.38D-6,          +0.00D-6,
     :               +2.27D-6,          -1.11D-6,
     :               -3.34D-6,          +0.00D-6,
     :               +3.34D-6,          +0.00D-6,
     :               -3.30D-6,          +0.01D-6,
     :               +3.31D-6,          +0.00D-6,
     :               +3.30D-6,          +0.00D-6,
     :               -3.30D-6,          +0.00D-6,
     :               -1.39D-6,          -1.91D-6,
     :               +3.30D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  331,  340 ) /
     :               +3.26D-6,          +0.00D-6,
     :               +3.26D-6,          +0.00D-6,
     :               +3.22D-6,          -0.04D-6,
     :               -3.26D-6,          +0.00D-6,
     :               +2.51D-6,          -0.64D-6,
     :               +3.14D-6,          +0.00D-6,
     :               -2.63D-6,          -0.48D-6,
     :               +3.10D-6,          +0.00D-6,
     :               -3.06D-6,          +0.00D-6,
     :               +2.94D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  341,  350 ) /
     :               +3.06D-6,          +0.00D-6,
     :               +0.00D-6,          +2.98D-6,
     :               +2.98D-6,          +0.00D-6,
     :               +2.07D-6,          +0.91D-6,
     :               -2.98D-6,          +0.00D-6,
     :               +2.94D-6,          +0.00D-6,
     :               -2.94D-6,          +0.00D-6,
     :               -2.94D-6,          +0.00D-6,
     :               -2.90D-6,          +0.00D-6,
     :               -0.56D-6,          -2.35D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  351,  360 ) /
     :               -1.47D-6,          +1.39D-6,
     :               +2.80D-6,          +0.00D-6,
     :               -2.74D-6,          +0.00D-6,
     :               -0.12D-6,          +2.63D-6,
     :               +2.15D-6,          -0.60D-6,
     :               -2.70D-6,          +0.00D-6,
     :               +1.79D-6,          -0.88D-6,
     :               -0.48D-6,          +2.19D-6,
     :               +0.44D-6,          +2.23D-6,
     :               +0.52D-6,          +2.07D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  361,  370 ) /
     :               -2.59D-6,          +0.00D-6,
     :               +2.55D-6,          +0.00D-6,
     :               -1.11D-6,          +1.43D-6,
     :               -2.51D-6,          +0.00D-6,
     :               -2.51D-6,          +0.00D-6,
     :               +2.51D-6,          +0.00D-6,
     :               +0.00D-6,          -2.50D-6,
     :               +2.47D-6,          +0.00D-6,
     :               +2.11D-6,          -0.36D-6,
     :               +1.67D-6,          +0.80D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  371,  380 ) /
     :               +2.46D-6,          +0.00D-6,
     :               -2.43D-6,          +0.00D-6,
     :               -2.39D-6,          +0.00D-6,
     :               -1.83D-6,          +0.56D-6,
     :               -0.44D-6,          -1.95D-6,
     :               +0.24D-6,          +2.15D-6,
     :               +2.39D-6,          +0.00D-6,
     :               +2.35D-6,          +0.00D-6,
     :               +2.27D-6,          +0.00D-6,
     :               -2.22D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  381,  390 ) /
     :               -1.03D-6,          -1.15D-6,
     :               +1.87D-6,          +0.32D-6,
     :               -0.32D-6,          -1.87D-6,
     :               +2.15D-6,          +0.00D-6,
     :               -0.80D-6,          +1.35D-6,
     :               +2.11D-6,          +0.00D-6,
     :               -2.11D-6,          +0.00D-6,
     :               -0.56D-6,          -1.55D-6,
     :               +2.11D-6,          +0.00D-6,
     :               -0.84D-6,          -1.27D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  391,  400 ) /
     :               -1.99D-6,          +0.12D-6,
     :               -0.24D-6,          +1.87D-6,
     :               -0.24D-6,          -1.87D-6,
     :               -2.03D-6,          +0.00D-6,
     :               +2.03D-6,          +0.00D-6,
     :               +2.03D-6,          +0.00D-6,
     :               +2.03D-6,          +0.00D-6,
     :               -0.40D-6,          +1.59D-6,
     :               +1.99D-6,          +0.00D-6,
     :               +1.95D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  401,  410 ) /
     :               +1.95D-6,          +0.00D-6,
     :               +1.91D-6,          +0.00D-6,
     :               +1.19D-6,          -0.72D-6,
     :               +1.87D-6,          +0.00D-6,
     :               +1.87D-6,          +0.00D-6,
     :               -1.27D-6,          +0.60D-6,
     :               +0.72D-6,          -1.15D-6,
     :               -0.99D-6,          +0.88D-6,
     :               +1.87D-6,          +0.00D-6,
     :               -1.87D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  411,  420 ) /
     :               -1.83D-6,          +0.00D-6,
     :               -1.79D-6,          +0.00D-6,
     :               -1.79D-6,          +0.00D-6,
     :               +1.79D-6,          +0.00D-6,
     :               +0.00D-6,          -1.79D-6,
     :               -1.79D-6,          +0.00D-6,
     :               -1.75D-6,          +0.00D-6,
     :               -1.75D-6,          +0.00D-6,
     :               +1.75D-6,          +0.00D-6,
     :               -1.47D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  421,  430 ) /
     :               -1.71D-6,          +0.00D-6,
     :               +1.71D-6,          +0.00D-6,
     :               +0.32D-6,          +1.39D-6,
     :               +0.28D-6,          -1.43D-6,
     :               -0.52D-6,          -1.19D-6,
     :               +1.67D-6,          +0.00D-6,
     :               -1.67D-6,          +0.00D-6,
     :               +0.76D-6,          -0.91D-6,
     :               -0.32D-6,          +1.35D-6,
     :               -1.39D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  431,  440 ) /
     :               +1.63D-6,          +0.00D-6,
     :               -1.59D-6,          +0.00D-6,
     :               +1.03D-6,          -0.56D-6,
     :               +1.59D-6,          +0.00D-6,
     :               +1.55D-6,          +0.00D-6,
     :               -0.28D-6,          -1.27D-6,
     :               -0.64D-6,          +0.91D-6,
     :               -0.32D-6,          -1.23D-6,
     :               -1.55D-6,          +0.00D-6,
     :               -1.51D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  441,  450 ) /
     :               +1.51D-6,          +0.00D-6,
     :               -1.51D-6,          +0.00D-6,
     :               +1.51D-6,          +0.00D-6,
     :               +1.47D-6,          +0.00D-6,
     :               +1.47D-6,          +0.00D-6,
     :               +0.95D-6,          -0.52D-6,
     :               +1.23D-6,          -0.24D-6,
     :               +0.60D-6,          +0.88D-6,
     :               -1.47D-6,          +0.00D-6,
     :               -1.43D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  451,  460 ) /
     :               +1.43D-6,          +0.00D-6,
     :               +1.43D-6,          +0.00D-6,
     :               -0.68D-6,          -0.76D-6,
     :               +0.95D-6,          -0.48D-6,
     :               -0.95D-6,          -0.48D-6,
     :               -1.19D-6,          -0.24D-6,
     :               +0.36D-6,          -1.07D-6,
     :               +0.95D-6,          +0.48D-6,
     :               +1.43D-6,          +0.00D-6,
     :               +1.39D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  461,  470 ) /
     :               +1.39D-6,          +0.00D-6,
     :               -1.39D-6,          +0.00D-6,
     :               -1.39D-6,          +0.00D-6,
     :               +0.00D-6,          +1.39D-6,
     :               -0.12D-6,          -1.27D-6,
     :               +0.56D-6,          +0.84D-6,
     :               -0.44D-6,          -0.95D-6,
     :               +0.32D-6,          -1.07D-6,
     :               +1.03D-6,          -0.36D-6,
     :               -0.28D-6,          +1.11D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  471,  480 ) /
     :               +0.44D-6,          +0.95D-6,
     :               -1.35D-6,          +0.00D-6,
     :               +0.88D-6,          +0.48D-6,
     :               -1.35D-6,          +0.00D-6,
     :               +1.35D-6,          +0.00D-6,
     :               +1.35D-6,          +0.00D-6,
     :               -1.31D-6,          +0.00D-6,
     :               +1.31D-6,          +0.00D-6,
     :               -1.19D-6,          -0.12D-6,
     :               +1.27D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  481,  490 ) /
     :               +0.40D-6,          -0.88D-6,
     :               +1.27D-6,          +0.00D-6,
     :               +1.27D-6,          +0.00D-6,
     :               -0.16D-6,          -1.11D-6,
     :               -0.84D-6,          +0.44D-6,
     :               +0.84D-6,          -0.44D-6,
     :               +0.84D-6,          -0.44D-6,
     :               -1.27D-6,          +0.00D-6,
     :               -1.27D-6,          +0.00D-6,
     :               +1.27D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  491,  500 ) /
     :               -0.44D-6,          -0.84D-6,
     :               +0.00D-6,          -1.27D-6,
     :               -1.27D-6,          +0.00D-6,
     :               -1.23D-6,          +0.00D-6,
     :               -1.23D-6,          +0.00D-6,
     :               +1.23D-6,          +0.00D-6,
     :               +0.00D-6,          +1.23D-6,
     :               -0.12D-6,          +1.11D-6,
     :               +1.22D-6,          +0.00D-6,
     :               +1.19D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  501,  510 ) /
     :               -0.24D-6,          +0.95D-6,
     :               -0.76D-6,          -0.44D-6,
     :               +0.91D-6,          +0.28D-6,
     :               +1.19D-6,          +0.00D-6,
     :               +1.19D-6,          +0.00D-6,
     :               +0.00D-6,          +1.19D-6,
     :               +1.15D-6,          +0.00D-6,
     :               +0.00D-6,          +1.15D-6,
     :               -1.15D-6,          +0.00D-6,
     :               +1.15D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  511,  520 ) /
     :               -1.15D-6,          +0.00D-6,
     :               +1.15D-6,          +0.00D-6,
     :               +1.15D-6,          +0.00D-6,
     :               -0.95D-6,          +0.20D-6,
     :               +0.24D-6,          +0.91D-6,
     :               -1.15D-6,          +0.00D-6,
     :               -1.12D-6,          +0.00D-6,
     :               -1.11D-6,          +0.00D-6,
     :               -1.11D-6,          +0.00D-6,
     :               +0.16D-6,          +0.95D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  521,  530 ) /
     :               -1.11D-6,          +0.00D-6,
     :               +1.11D-6,          +0.00D-6,
     :               +0.20D-6,          -0.91D-6,
     :               -0.72D-6,          -0.40D-6,
     :               -1.11D-6,          +0.00D-6,
     :               -1.11D-6,          +0.00D-6,
     :               +1.07D-6,          +0.00D-6,
     :               -1.07D-6,          +0.00D-6,
     :               +0.76D-6,          -0.32D-6,
     :               +0.00D-6,          -1.07D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  531,  540 ) /
     :               +1.07D-6,          +0.00D-6,
     :               +1.07D-6,          +0.00D-6,
     :               -1.07D-6,          +0.00D-6,
     :               +1.07D-6,          +0.00D-6,
     :               -0.84D-6,          -0.24D-6,
     :               +0.00D-6,          -1.03D-6,
     :               +1.03D-6,          +0.00D-6,
     :               -1.03D-6,          +0.00D-6,
     :               -0.24D-6,          +0.80D-6,
     :               +0.20D-6,          +0.84D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  541,  550 ) /
     :               -1.03D-6,          +0.00D-6,
     :               -1.03D-6,          +0.00D-6,
     :               -0.99D-6,          +0.00D-6,
     :               +0.24D-6,          +0.76D-6,
     :               -0.99D-6,          +0.00D-6,
     :               -0.16D-6,          +0.84D-6,
     :               -0.99D-6,          +0.00D-6,
     :               -0.64D-6,          +0.36D-6,
     :               +0.99D-6,          +0.00D-6,
     :               +0.36D-6,          -0.64D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  551,  560 ) /
     :               -0.95D-6,          +0.00D-6,
     :               -0.95D-6,          +0.00D-6,
     :               +0.00D-6,          +0.95D-6,
     :               +0.64D-6,          +0.32D-6,
     :               +0.00D-6,          -0.95D-6,
     :               +0.84D-6,          +0.12D-6,
     :               +0.20D-6,          +0.76D-6,
     :               -0.95D-6,          +0.00D-6,
     :               +0.95D-6,          +0.00D-6,
     :               -0.95D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  561,  570 ) /
     :               +0.00D-6,          +0.92D-6,
     :               +0.91D-6,          +0.00D-6,
     :               +0.91D-6,          +0.00D-6,
     :               +0.40D-6,          +0.52D-6,
     :               -0.91D-6,          +0.00D-6,
     :               -0.56D-6,          +0.36D-6,
     :               +0.44D-6,          -0.48D-6,
     :               -0.91D-6,          +0.00D-6,
     :               -0.91D-6,          +0.00D-6,
     :               -0.36D-6,          -0.56D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  571,  580 ) /
     :               +0.91D-6,          +0.00D-6,
     :               -0.88D-6,          +0.00D-6,
     :               -0.88D-6,          +0.00D-6,
     :               +0.60D-6,          -0.28D-6,
     :               +0.88D-6,          +0.00D-6,
     :               +0.36D-6,          -0.52D-6,
     :               -0.52D-6,          +0.36D-6,
     :               +0.52D-6,          +0.36D-6,
     :               +0.00D-6,          +0.88D-6,
     :               +0.56D-6,          +0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  581,  590 ) /
     :               +0.64D-6,          -0.24D-6,
     :               +0.88D-6,          +0.00D-6,
     :               +0.88D-6,          +0.00D-6,
     :               +0.88D-6,          +0.00D-6,
     :               +0.84D-6,          +0.00D-6,
     :               -0.68D-6,          -0.16D-6,
     :               +0.84D-6,          +0.00D-6,
     :               +0.56D-6,          +0.28D-6,
     :               -0.16D-6,          +0.68D-6,
     :               +0.64D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  591,  600 ) /
     :               +0.16D-6,          +0.68D-6,
     :               +0.72D-6,          -0.12D-6,
     :               -0.83D-6,          +0.00D-6,
     :               -0.80D-6,          +0.00D-6,
     :               +0.80D-6,          +0.00D-6,
     :               -0.80D-6,          +0.00D-6,
     :               +0.28D-6,          +0.52D-6,
     :               +0.68D-6,          -0.12D-6,
     :               +0.00D-6,          -0.80D-6,
     :               -0.32D-6,          +0.48D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  601,  610 ) /
     :               +0.36D-6,          -0.44D-6,
     :               -0.36D-6,          -0.44D-6,
     :               -0.80D-6,          +0.00D-6,
     :               +0.79D-6,          +0.00D-6,
     :               +0.74D-6,          -0.04D-6,
     :               -0.76D-6,          +0.00D-6,
     :               +0.00D-6,          +0.76D-6,
     :               +0.16D-6,          +0.60D-6,
     :               -0.76D-6,          +0.00D-6,
     :               -0.76D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  611,  620 ) /
     :               +0.76D-6,          +0.00D-6,
     :               -0.76D-6,          +0.00D-6,
     :               +0.76D-6,          +0.00D-6,
     :               +0.12D-6,          +0.64D-6,
     :               +0.76D-6,          +0.00D-6,
     :               +0.00D-6,          +0.76D-6,
     :               +0.76D-6,          +0.00D-6,
     :               +0.64D-6,          -0.12D-6,
     :               +0.16D-6,          -0.60D-6,
     :               +0.76D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  621,  630 ) /
     :               +0.00D-6,          -0.76D-6,
     :               +0.28D-6,          -0.48D-6,
     :               +0.32D-6,          +0.44D-6,
     :               -0.76D-6,          +0.00D-6,
     :               +0.72D-6,          +0.00D-6,
     :               +0.72D-6,          +0.00D-6,
     :               +0.48D-6,          -0.24D-6,
     :               -0.72D-6,          +0.00D-6,
     :               +0.72D-6,          +0.00D-6,
     :               -0.72D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  631,  640 ) /
     :               -0.72D-6,          +0.00D-6,
     :               -0.71D-6,          +0.00D-6,
     :               -0.68D-6,          +0.00D-6,
     :               -0.68D-6,          +0.00D-6,
     :               +0.68D-6,          +0.00D-6,
     :               +0.68D-6,          +0.00D-6,
     :               +0.68D-6,          +0.00D-6,
     :               -0.68D-6,          +0.00D-6,
     :               +0.56D-6,          -0.12D-6,
     :               -0.68D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  641,  650 ) /
     :               -0.68D-6,          +0.00D-6,
     :               +0.20D-6,          +0.48D-6,
     :               -0.44D-6,          -0.24D-6,
     :               -0.68D-6,          +0.00D-6,
     :               +0.64D-6,          +0.00D-6,
     :               +0.64D-6,          +0.00D-6,
     :               -0.64D-6,          +0.00D-6,
     :               +0.64D-6,          +0.00D-6,
     :               -0.64D-6,          +0.00D-6,
     :               -0.12D-6,          +0.52D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  651,  660 ) /
     :               -0.12D-6,          -0.52D-6,
     :               -0.16D-6,          -0.48D-6,
     :               -0.20D-6,          -0.44D-6,
     :               -0.44D-6,          +0.20D-6,
     :               -0.44D-6,          +0.20D-6,
     :               +0.24D-6,          -0.40D-6,
     :               -0.20D-6,          -0.44D-6,
     :               -0.64D-6,          +0.00D-6,
     :               +0.40D-6,          -0.24D-6,
     :               -0.64D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  661,  670 ) /
     :               +0.64D-6,          +0.00D-6,
     :               -0.63D-6,          +0.00D-6,
     :               -0.60D-6,          +0.00D-6,
     :               +0.00D-6,          +0.60D-6,
     :               -0.60D-6,          +0.00D-6,
     :               -0.60D-6,          +0.00D-6,
     :               +0.60D-6,          +0.00D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.24D-6,          -0.36D-6,
     :               +0.12D-6,          +0.48D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  671,  680 ) /
     :               +0.48D-6,          -0.12D-6,
     :               +0.12D-6,          +0.48D-6,
     :               +0.24D-6,          -0.36D-6,
     :               +0.36D-6,          +0.24D-6,
     :               +0.12D-6,          +0.48D-6,
     :               +0.44D-6,          +0.16D-6,
     :               -0.60D-6,          +0.00D-6,
     :               -0.60D-6,          +0.00D-6,
     :               +0.60D-6,          +0.00D-6,
     :               +0.00D-6,          +0.60D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  681,  690 ) /
     :               +0.59D-6,          +0.00D-6,
     :               -0.56D-6,          +0.00D-6,
     :               -0.44D-6,          -0.12D-6,
     :               +0.56D-6,          +0.00D-6,
     :               +0.00D-6,          +0.56D-6,
     :               -0.56D-6,          +0.00D-6,
     :               -0.56D-6,          +0.00D-6,
     :               +0.56D-6,          +0.00D-6,
     :               -0.56D-6,          +0.00D-6,
     :               +0.16D-6,          +0.40D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  691,  700 ) /
     :               +0.44D-6,          -0.12D-6,
     :               +0.20D-6,          -0.36D-6,
     :               -0.36D-6,          -0.20D-6,
     :               -0.56D-6,          +0.00D-6,
     :               +0.55D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               +0.16D-6,          +0.36D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  701,  710 ) /
     :               -0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               +0.00D-6,          -0.52D-6,
     :               +0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               +0.12D-6,          +0.40D-6,
     :               +0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  711,  720 ) /
     :               +0.00D-6,          -0.52D-6,
     :               +0.52D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               -0.51D-6,          +0.00D-6,
     :               -0.51D-6,          +0.00D-6,
     :               +0.48D-6,          +0.00D-6,
     :               +0.48D-6,          +0.00D-6,
     :               -0.16D-6,          +0.32D-6,
     :               -0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  721,  730 ) /
     :               +0.48D-6,          +0.00D-6,
     :               +0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6,
     :               -0.12D-6,          -0.36D-6,
     :               -0.32D-6,          +0.16D-6,
     :               +0.32D-6,          -0.16D-6,
     :               -0.12D-6,          -0.36D-6,
     :               +0.16D-6,          +0.32D-6,
     :               +0.20D-6,          -0.28D-6,
     :               -0.20D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  731,  740 ) /
     :               -0.36D-6,          +0.12D-6,
     :               -0.48D-6,          +0.00D-6,
     :               +0.32D-6,          -0.16D-6,
     :               +0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6,
     :               +0.00D-6,          -0.48D-6,
     :               +0.48D-6,          +0.00D-6,
     :               -0.48D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  741,  750 ) /
     :               -0.48D-6,          +0.00D-6,
     :               +0.00D-6,          +0.48D-6,
     :               +0.44D-6,          +0.00D-6,
     :               -0.32D-6,          -0.12D-6,
     :               -0.44D-6,          +0.00D-6,
     :               +0.20D-6,          -0.24D-6,
     :               +0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               +0.44D-6,          +0.00D-6,
     :               +0.20D-6,          -0.24D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  751,  760 ) /
     :               +0.12D-6,          +0.32D-6,
     :               -0.20D-6,          +0.24D-6,
     :               +0.32D-6,          -0.12D-6,
     :               +0.00D-6,          +0.44D-6,
     :               +0.00D-6,          +0.44D-6,
     :               +0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               +0.44D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  761,  770 ) /
     :               +0.44D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.24D-6,          +0.16D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.12D-6,          +0.28D-6,
     :               +0.40D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  771,  780 ) /
     :               -0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6,
     :               -0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.20D-6,          -0.20D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  781,  790 ) /
     :               -0.12D-6,          -0.28D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          +0.40D-6,
     :               -0.20D-6,          -0.16D-6,
     :               +0.36D-6,          +0.00D-6,
     :               +0.36D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  791,  800 ) /
     :               +0.24D-6,          -0.12D-6,
     :               +0.20D-6,          -0.16D-6,
     :               +0.00D-6,          +0.36D-6,
     :               +0.00D-6,          +0.36D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.12D-6,          +0.24D-6,
     :               -0.36D-6,          +0.00D-6,
     :               -0.36D-6,          +0.00D-6,
     :               -0.36D-6,          +0.00D-6,
     :               -0.36D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  801,  810 ) /
     :               +0.36D-6,          +0.00D-6,
     :               +0.00D-6,          +0.36D-6,
     :               +0.00D-6,          +0.36D-6,
     :               +0.00D-6,          +0.36D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.00D-6,          +0.36D-6,
     :               +0.12D-6,          -0.24D-6,
     :               -0.24D-6,          +0.12D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.00D-6,          +0.36D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  811,  820 ) /
     :               +0.36D-6,          +0.00D-6,
     :               +0.24D-6,          -0.12D-6,
     :               +0.00D-6,          -0.36D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.36D-6,          +0.00D-6,
     :               +0.36D-6,          +0.00D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.36D-6,          +0.00D-6,
     :               -0.13D-6,          +0.22D-6,
     :               -0.32D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  821,  830 ) /
     :               -0.32D-6,          +0.00D-6,
     :               +0.32D-6,          +0.00D-6,
     :               -0.20D-6,          -0.12D-6,
     :               +0.32D-6,          +0.00D-6,
     :               +0.12D-6,          +0.20D-6,
     :               -0.32D-6,          +0.00D-6,
     :               +0.32D-6,          +0.00D-6,
     :               -0.32D-6,          +0.00D-6,
     :               -0.32D-6,          +0.00D-6,
     :               +0.00D-6,          -0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  831,  840 ) /
     :               +0.32D-6,          +0.00D-6,
     :               +0.32D-6,          +0.00D-6,
     :               +0.12D-6,          -0.20D-6,
     :               -0.32D-6,          +0.00D-6,
     :               +0.00D-6,          -0.32D-6,
     :               +0.32D-6,          +0.00D-6,
     :               +0.00D-6,          +0.32D-6,
     :               +0.00D-6,          -0.32D-6,
     :               +0.00D-6,          -0.32D-6,
     :               +0.32D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  841,  850 ) /
     :               -0.32D-6,          +0.00D-6,
     :               +0.00D-6,          +0.32D-6,
     :               +0.32D-6,          +0.00D-6,
     :               +0.00D-6,          +0.32D-6,
     :               +0.00D-6,          -0.32D-6,
     :               +0.32D-6,          +0.00D-6,
     :               -0.16D-6,          +0.16D-6,
     :               -0.16D-6,          +0.16D-6,
     :               +0.00D-6,          +0.32D-6,
     :               +0.20D-6,          +0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  851,  860 ) /
     :               +0.20D-6,          +0.12D-6,
     :               -0.20D-6,          +0.12D-6,
     :               +0.12D-6,          +0.20D-6,
     :               +0.12D-6,          -0.20D-6,
     :               +0.00D-6,          +0.32D-6,
     :               -0.32D-6,          +0.00D-6,
     :               +0.32D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  861,  870 ) /
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.16D-6,          +0.12D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               -0.12D-6,          -0.16D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  871,  880 ) /
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.00D-6,          +0.28D-6,
     :               +0.00D-6,          +0.28D-6,
     :               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  881,  890 ) /
     :               -0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.12D-6,          -0.16D-6,
     :               +0.00D-6,          +0.28D-6,
     :               +0.00D-6,          -0.28D-6,
     :               +0.12D-6,          -0.16D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.00D-6,          -0.28D-6,
     :               +0.00D-6,          +0.28D-6,
     :               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  891,  900 ) /
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.12D-6,          -0.16D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.28D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  901,  910 ) /
     :               +0.00D-6,          -0.28D-6,
     :               +0.28D-6,          +0.00D-6,
     :               +0.00D-6,          +0.24D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  911,  920 ) /
     :               +0.24D-6,          +0.00D-6,
     :               +0.00D-6,          -0.24D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  921,  930 ) /
     :               +0.24D-6,          +0.00D-6,
     :               +0.00D-6,          -0.24D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.00D-6,          +0.24D-6,
     :               +0.12D-6,          -0.12D-6,
     :               +0.00D-6,          -0.24D-6,
     :               -0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  931,  940 ) /
     :               +0.00D-6,          +0.24D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.00D-6,          -0.24D-6,
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  941,  950 ) /
     :               +0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               -0.24D-6,          +0.00D-6,
     :               +0.00D-6,          +0.24D-6,
     :               +0.00D-6,          -0.24D-6,
     :               +0.24D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  951,  960 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  961,  970 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  971,  980 ) /
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  981,  990 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  991, 1000 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1001, 1010 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1011, 1020 ) /
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1021, 1030 ) /
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1031, 1040 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1041, 1050 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.04D-6,          +0.12D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1051, 1060 ) /
     :               +0.00D-6,          +0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1061, 1070 ) /
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6 /

      DATA ( ( SX0(I,J), I=1,2), J = 1071, 1080 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1081, 1090 ) /
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.00D-6,          +0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1091, 1100 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.00D-6,          -0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1101, 1110 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          +0.16D-6,
     :               +0.00D-6,          +0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1111, 1120 ) /
     :               +0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               -0.16D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6,
     :               +0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1121, 1130 ) /
     :               +0.16D-6,          +0.00D-6,
     :               -0.15D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1131, 1140 ) /
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1141, 1150 ) /
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1151, 1160 ) /
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1161, 1170 ) /
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1171, 1180 ) /
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.08D-6,          +0.04D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1181, 1190 ) /
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1191, 1200 ) /
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1201, 1210 ) /
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1211, 1220 ) /
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1221, 1230 ) /
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1231, 1240 ) /
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1241, 1250 ) /
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1251, 1260 ) /
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1261, 1270 ) /
     :               +0.00D-6,          +0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1271, 1280 ) /
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1281, 1290 ) /
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1291, 1300 ) /
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1301,  NX0 ) /
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.11D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^1 terms
      DATA ( ( SX1(I,J), I=1,2), J =    1,   10 ) /
     :            -3328.48D-6,     +205833.15D-6,
     :             +197.53D-6,      +12814.01D-6,
     :              +41.19D-6,       +2187.91D-6,
     :              -35.85D-6,       -2004.36D-6,
     :              +59.15D-6,        +501.82D-6,
     :               -5.82D-6,        +448.76D-6,
     :             -179.56D-6,        +164.33D-6,
     :               +5.67D-6,        +288.49D-6,
     :              +23.85D-6,        -214.50D-6,
     :               +2.87D-6,        -154.91D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   11,   20 ) /
     :               +2.14D-6,        -119.21D-6,
     :               +1.17D-6,         -74.33D-6,
     :               +1.47D-6,         +70.31D-6,
     :               -0.42D-6,         +58.94D-6,
     :               -0.95D-6,         +57.12D-6,
     :               -1.08D-6,         -54.19D-6,
     :               +0.92D-6,         +36.78D-6,
     :               +0.68D-6,         -31.01D-6,
     :               +0.74D-6,         +29.60D-6,
     :               -0.60D-6,         -27.59D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   21,   30 ) /
     :              -11.11D-6,         -15.07D-6,
     :               -0.40D-6,         -24.05D-6,
     :               -0.81D-6,         +19.06D-6,
     :               +3.18D-6,         +15.32D-6,
     :               -0.08D-6,         -17.90D-6,
     :               -0.16D-6,         +15.55D-6,
     :               -0.77D-6,         +14.40D-6,
     :               -0.25D-6,         +11.67D-6,
     :               +6.18D-6,          +3.58D-6,
     :               -1.00D-6,          -7.27D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   31,   40 ) /
     :               -0.99D-6,          +6.87D-6,
     :               -0.27D-6,          +7.49D-6,
     :               -0.30D-6,          +7.31D-6,
     :               +0.20D-6,          +7.30D-6,
     :               +0.33D-6,          +6.80D-6,
     :               +0.27D-6,          -6.81D-6,
     :               +0.35D-6,          +6.08D-6,
     :               +0.35D-6,          +6.09D-6,
     :               -0.14D-6,          -6.19D-6,
     :               +0.14D-6,          +6.02D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   41,   50 ) /
     :               +2.69D-6,          -2.76D-6,
     :               -0.08D-6,          -4.93D-6,
     :               +2.85D-6,          -1.77D-6,
     :               -0.07D-6,          -4.27D-6,
     :               -3.71D-6,          +0.38D-6,
     :               +3.75D-6,          +0.04D-6,
     :               -0.82D-6,          -2.73D-6,
     :               -0.06D-6,          +2.93D-6,
     :               -0.04D-6,          +2.83D-6,
     :               +0.08D-6,          +2.75D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   51,   60 ) /
     :               +0.07D-6,          +2.75D-6,
     :               -0.07D-6,          +2.70D-6,
     :               -0.07D-6,          +2.52D-6,
     :               -0.05D-6,          -2.53D-6,
     :               -0.04D-6,          +2.40D-6,
     :               -0.06D-6,          -2.37D-6,
     :               +0.69D-6,          -1.45D-6,
     :               -0.04D-6,          +2.00D-6,
     :               +1.99D-6,          +0.02D-6,
     :               -0.94D-6,          +1.07D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   61,   70 ) /
     :               +0.04D-6,          +1.91D-6,
     :               -0.58D-6,          -1.36D-6,
     :               -0.51D-6,          -1.25D-6,
     :               -0.04D-6,          -1.59D-6,
     :               +0.39D-6,          -1.23D-6,
     :               +0.03D-6,          -1.57D-6,
     :               -0.03D-6,          +1.50D-6,
     :               +0.04D-6,          +1.48D-6,
     :               -0.04D-6,          +1.45D-6,
     :               +0.02D-6,          -1.36D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   71,   80 ) /
     :               +0.03D-6,          -1.32D-6,
     :               -0.03D-6,          -1.24D-6,
     :               -0.02D-6,          -1.18D-6,
     :               -0.03D-6,          +1.16D-6,
     :               +0.02D-6,          +1.13D-6,
     :               +0.04D-6,          -1.11D-6,
     :               +0.02D-6,          +1.11D-6,
     :               +0.03D-6,          -1.10D-6,
     :               +0.03D-6,          +1.04D-6,
     :               -0.51D-6,          +0.56D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   81,   90 ) /
     :               +0.02D-6,          -0.98D-6,
     :               -0.02D-6,          -0.94D-6,
     :               -0.02D-6,          -0.89D-6,
     :               -0.02D-6,          -0.88D-6,
     :               +0.31D-6,          +0.60D-6,
     :               +0.02D-6,          -0.87D-6,
     :               -0.02D-6,          -0.87D-6,
     :               -0.01D-6,          +0.83D-6,
     :               -0.02D-6,          +0.77D-6,
     :               +0.42D-6,          -0.36D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   91,  100 ) /
     :               -0.01D-6,          -0.73D-6,
     :               +0.01D-6,          +0.71D-6,
     :               +0.01D-6,          +0.68D-6,
     :               +0.02D-6,          +0.66D-6,
     :               -0.01D-6,          -0.62D-6,
     :               -0.01D-6,          +0.62D-6,
     :               -0.58D-6,          -0.03D-6,
     :               -0.01D-6,          +0.58D-6,
     :               +0.44D-6,          +0.14D-6,
     :               +0.02D-6,          +0.56D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  101,  110 ) /
     :               +0.01D-6,          -0.57D-6,
     :               -0.13D-6,          -0.45D-6,
     :               +0.01D-6,          +0.56D-6,
     :               +0.01D-6,          -0.55D-6,
     :               +0.01D-6,          +0.55D-6,
     :               -0.52D-6,          +0.03D-6,
     :               -0.01D-6,          +0.54D-6,
     :               -0.01D-6,          -0.51D-6,
     :               -0.41D-6,          -0.11D-6,
     :               -0.01D-6,          +0.50D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  111,  120 ) /
     :               +0.01D-6,          +0.48D-6,
     :               +0.45D-6,          -0.04D-6,
     :               +0.01D-6,          -0.48D-6,
     :               +0.01D-6,          +0.46D-6,
     :               -0.23D-6,          +0.24D-6,
     :               +0.01D-6,          +0.46D-6,
     :               +0.35D-6,          -0.11D-6,
     :               +0.01D-6,          +0.45D-6,
     :               +0.01D-6,          -0.45D-6,
     :               +0.00D-6,          -0.45D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  121,  130 ) /
     :               -0.01D-6,          +0.44D-6,
     :               +0.35D-6,          +0.09D-6,
     :               +0.01D-6,          +0.42D-6,
     :               -0.01D-6,          -0.41D-6,
     :               +0.09D-6,          -0.33D-6,
     :               +0.00D-6,          +0.41D-6,
     :               +0.01D-6,          +0.40D-6,
     :               -0.01D-6,          -0.39D-6,
     :               -0.39D-6,          -0.01D-6,
     :               +0.01D-6,          -0.39D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  131,  140 ) /
     :               -0.01D-6,          +0.38D-6,
     :               +0.32D-6,          -0.07D-6,
     :               -0.01D-6,          +0.36D-6,
     :               -0.01D-6,          -0.36D-6,
     :               +0.01D-6,          -0.34D-6,
     :               +0.01D-6,          -0.34D-6,
     :               +0.01D-6,          +0.33D-6,
     :               -0.01D-6,          -0.32D-6,
     :               +0.01D-6,          +0.32D-6,
     :               -0.01D-6,          -0.32D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  141,  150 ) /
     :               -0.01D-6,          -0.31D-6,
     :               -0.31D-6,          +0.00D-6,
     :               -0.07D-6,          -0.24D-6,
     :               +0.10D-6,          -0.21D-6,
     :               -0.01D-6,          -0.30D-6,
     :               -0.01D-6,          +0.29D-6,
     :               -0.01D-6,          -0.29D-6,
     :               +0.00D-6,          +0.29D-6,
     :               +0.23D-6,          +0.06D-6,
     :               +0.26D-6,          +0.02D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  151,  160 ) /
     :               +0.00D-6,          -0.27D-6,
     :               +0.25D-6,          +0.02D-6,
     :               +0.09D-6,          -0.18D-6,
     :               +0.01D-6,          +0.25D-6,
     :               +0.14D-6,          -0.11D-6,
     :               +0.00D-6,          -0.25D-6,
     :               +0.01D-6,          +0.24D-6,
     :               -0.01D-6,          -0.24D-6,
     :               +0.00D-6,          +0.23D-6,
     :               +0.01D-6,          +0.23D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  161,  170 ) /
     :               -0.01D-6,          -0.23D-6,
     :               +0.00D-6,          -0.23D-6,
     :               +0.00D-6,          -0.22D-6,
     :               +0.00D-6,          +0.21D-6,
     :               +0.01D-6,          +0.21D-6,
     :               -0.17D-6,          +0.03D-6,
     :               -0.17D-6,          +0.03D-6,
     :               +0.00D-6,          -0.19D-6,
     :               +0.14D-6,          -0.06D-6,
     :               +0.03D-6,          -0.17D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  171,  180 ) /
     :               -0.13D-6,          +0.06D-6,
     :               +0.00D-6,          +0.19D-6,
     :               +0.00D-6,          +0.19D-6,
     :               -0.06D-6,          -0.13D-6,
     :               +0.00D-6,          +0.18D-6,
     :               -0.09D-6,          -0.09D-6,
     :               +0.10D-6,          -0.09D-6,
     :               +0.06D-6,          +0.12D-6,
     :               +0.00D-6,          +0.18D-6,
     :               +0.00D-6,          -0.18D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  181,  190 ) /
     :               +0.00D-6,          +0.17D-6,
     :               -0.03D-6,          +0.15D-6,
     :               -0.01D-6,          -0.16D-6,
     :               +0.00D-6,          +0.17D-6,
     :               +0.00D-6,          -0.17D-6,
     :               +0.11D-6,          +0.06D-6,
     :               +0.00D-6,          -0.17D-6,
     :               -0.08D-6,          +0.09D-6,
     :               -0.17D-6,          +0.00D-6,
     :               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  191,  200 ) /
     :               +0.01D-6,          +0.15D-6,
     :               -0.13D-6,          -0.03D-6,
     :               +0.00D-6,          +0.15D-6,
     :               +0.00D-6,          +0.15D-6,
     :               -0.13D-6,          +0.03D-6,
     :               +0.10D-6,          -0.06D-6,
     :               -0.07D-6,          +0.08D-6,
     :               -0.09D-6,          -0.06D-6,
     :               +0.00D-6,          +0.15D-6,
     :               -0.07D-6,          -0.08D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  201,  210 ) /
     :               +0.00D-6,          -0.14D-6,
     :               +0.02D-6,          +0.12D-6,
     :               +0.07D-6,          +0.08D-6,
     :               -0.03D-6,          -0.11D-6,
     :               -0.01D-6,          -0.14D-6,
     :               +0.00D-6,          -0.14D-6,
     :               +0.02D-6,          -0.12D-6,
     :               +0.00D-6,          -0.14D-6,
     :               +0.00D-6,          +0.14D-6,
     :               +0.00D-6,          +0.14D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  211,  220 ) /
     :               +0.00D-6,          +0.13D-6,
     :               +0.08D-6,          -0.06D-6,
     :               +0.00D-6,          +0.13D-6,
     :               +0.00D-6,          +0.13D-6,
     :               +0.01D-6,          +0.13D-6,
     :               +0.00D-6,          +0.13D-6,
     :               +0.00D-6,          +0.13D-6,
     :               -0.02D-6,          -0.11D-6,
     :               +0.08D-6,          -0.04D-6,
     :               +0.00D-6,          +0.13D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  221,  230 ) /
     :               +0.00D-6,          +0.13D-6,
     :               +0.01D-6,          -0.12D-6,
     :               +0.00D-6,          +0.12D-6,
     :               -0.02D-6,          -0.11D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.04D-6,          +0.08D-6,
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          +0.12D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  231,  240 ) /
     :               +0.00D-6,          -0.12D-6,
     :               +0.00D-6,          -0.11D-6,
     :               +0.03D-6,          -0.09D-6,
     :               +0.00D-6,          +0.11D-6,
     :               -0.11D-6,          +0.00D-6,
     :               +0.00D-6,          +0.11D-6,
     :               +0.00D-6,          -0.11D-6,
     :               +0.07D-6,          +0.05D-6,
     :               +0.11D-6,          +0.00D-6,
     :               +0.00D-6,          -0.11D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  241,  250 ) /
     :               +0.00D-6,          -0.11D-6,
     :               +0.02D-6,          -0.09D-6,
     :               +0.00D-6,          +0.11D-6,
     :               +0.02D-6,          +0.09D-6,
     :               +0.00D-6,          -0.11D-6,
     :               +0.00D-6,          +0.11D-6,
     :               -0.08D-6,          -0.02D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               -0.03D-6,          -0.07D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  251,  NX1 ) /
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          -0.10D-6 /

*  Sine and cosine coefficients for t^2 terms
      DATA ( ( SX2(I,J), I=1,2), J =    1,   10 ) /
     :            +2038.00D-6,         +82.26D-6,
     :             +155.75D-6,          -2.70D-6,
     :              +26.92D-6,          -0.45D-6,
     :              -24.43D-6,          +0.46D-6,
     :              -17.36D-6,          -0.50D-6,
     :               -8.41D-6,          +0.01D-6,
     :               +6.08D-6,          -1.36D-6,
     :               +4.59D-6,          +0.17D-6,
     :               +3.57D-6,          -0.06D-6,
     :               +2.54D-6,          +0.60D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   11,   20 ) /
     :               +1.86D-6,          +0.00D-6,
     :               -1.52D-6,          -0.07D-6,
     :               +1.46D-6,          +0.04D-6,
     :               -0.75D-6,          -0.02D-6,
     :               -0.75D-6,          +0.00D-6,
     :               -0.71D-6,          -0.01D-6,
     :               -0.69D-6,          +0.02D-6,
     :               +0.61D-6,          +0.02D-6,
     :               +0.54D-6,          -0.04D-6,
     :               -0.56D-6,          +0.00D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   21,   30 ) /
     :               +0.46D-6,          -0.02D-6,
     :               +0.38D-6,          -0.01D-6,
     :               +0.37D-6,          -0.02D-6,
     :               -0.34D-6,          +0.01D-6,
     :               -0.35D-6,          +0.00D-6,
     :               -0.31D-6,          +0.00D-6,
     :               +0.19D-6,          -0.09D-6,
     :               +0.26D-6,          +0.00D-6,
     :               +0.24D-6,          -0.01D-6,
     :               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   31,  NX2 ) /
     :               +0.18D-6,          -0.01D-6,
     :               +0.17D-6,          +0.00D-6,
     :               +0.15D-6,          +0.01D-6,
     :               -0.15D-6,          +0.00D-6,
     :               -0.13D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^3 terms
      DATA ( ( SX3(I,J), I=1,2), J =    1,  NX3 ) /
     :               +1.76D-6,         -20.39D-6,
     :               +0.00D-6,          -1.27D-6,
     :               +0.00D-6,          -0.22D-6,
     :               +0.00D-6,          +0.20D-6 /

*  Sine and cosine coefficients for t^4 terms
      DATA ( ( SX4(I,J), I=1,2), J =    1,  NX4 ) /
     :               -0.10D-6,          -0.02D-6 /

*  ----------------------------------
*  The series for Y: numerical values
*  ----------------------------------

*  Polynomial coefficients
      DATA YP /      -6950.78D-6,
     :              -25381.99D-6,
     :           -22407250.99D-6,
     :                1842.28D-6,
     :                1113.06D-6,
     :                   0.99D-6 /

*  Argument coefficients for t^0 terms
      DATA ( ( KY0(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   21,   30 ) /
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   31,   40 ) /
     :  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   41,   50 ) /
     :  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1,
     :  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   51,   60 ) /
     :  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2,
     :  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   61,   70 ) /
     :  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   71,   80 ) /
     :  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   81,   90 ) /
     :  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1,
     :  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   91,  100 ) /
     :  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2,
     :  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  101,  110 ) /
     :  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
     :  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2,
     :  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  111,  120 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
     :  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2,
     :  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  121,  130 ) /
     :  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
     :  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  131,  140 ) /
     :  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
     :  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
     :  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  141,  150 ) /
     :  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1,
     :  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  151,  160 ) /
     :  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  161,  170 ) /
     :  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0,
     :  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  171,  180 ) /
     :  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2,
     :  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
     :  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  181,  190 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2,
     :  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1,
     :  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  191,  200 ) /
     :  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2,
     :  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2,
     :  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2,
     :  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
     :  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  201,  210 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
     :  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
     :  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  211,  220 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
     :  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1,
     :  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  221,  230 ) /
     :  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2,
     :  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2,
     :  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  231,  240 ) /
     :  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0,
     :  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  241,  250 ) /
     :  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  251,  260 ) /
     :  0,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0,
     :  3,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0,
     :  2, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  261,  270 ) /
     :  0,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -1,
     :  2, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2,
     :  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2,
     :  1,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  271,  280 ) /
     :  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, -1,
     :  1, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  281,  290 ) /
     :  2,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,
     :  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  291,  300 ) /
     :  1,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -2,
     :  2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,
     :  1,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  301,  310 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0, -1,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  311,  320 ) /
     :  0,  0,  0,  0,  0,  0,  0,  6,-16,  4,  5,  0,  0, -2,
     :  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -1,
     :  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  321,  330 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, -1,
     :  0,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  331,  340 ) /
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2,
     :  1,  0, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  341,  350 ) /
     :  1, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -1,
     :  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  351,  360 ) /
     :  2, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0,
     :  2,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2,
     :  0,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  361,  370 ) /
     :  2,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -1,
     :  2,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0,
     :  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  371,  380 ) /
     :  0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,
     :  2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  381,  390 ) /
     :  1,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0,
     :  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  391,  400 ) /
     :  0,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2,
     :  0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0, -2,
     :  1,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  401,  410 ) /
     :  0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  411,  420 ) /
     :  1,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  421,  430 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1,
     :  3,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2,
     :  0,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0,
     :  2,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  431,  440 ) /
     :  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2,
     :  1,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  441,  450 ) /
     :  2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0,
     :  0,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  451,  460 ) /
     :  1, -2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0,
     :  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  461,  470 ) /
     :  1, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -9,  4,  0,  0,  0,  0, -2,
     :  1,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  471,  480 ) /
     :  2,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  481,  490 ) /
     :  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -2,
     :  5,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2,
     :  1, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  491,  500 ) /
     :  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  4, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  501,  510 ) /
     :  0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2,
     :  0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0,
     :  0,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  511,  520 ) /
     :  1,  0, -2, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0, -6,  8,  0,  0,  0,  0,  0,  0,
     :  0,  1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  521,  530 ) /
     :  0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0,
     :  0,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6,-15,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  531,  540 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0, -2,
     :  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  541,  550 ) /
     :  2, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  0,  0,  2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1, -1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  551,  560 ) /
     :  1,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -1,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  0,  5,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  561,  570 ) /
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  3,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0,
     :  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  571,  580 ) /
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1, -5,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1,
     :  4,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  581,  590 ) /
     :  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -5,  6,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  591,  600 ) /
     :  2,  0, -1, -1, -1,  0,  0,  3, -7,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2,
     :  3, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  601,  610 ) /
     :  3, -1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  6,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9,-12,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  611,  620 ) /
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2,
     :  0,  0,  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  621,  630 ) /
     :  0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0,
     :  1,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  631,  640 ) /
     :  4,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0,
     :  1,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  641,  650 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2,
     :  2, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2,
     :  5,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  651,  660 ) /
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  661,  670 ) /
     :  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,
     :  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  671,  680 ) /
     :  1, -2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  681,  690 ) /
     :  0,  0,  2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  2,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1, -1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  1, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  691,  700 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0,
     :  2,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -6,  8,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0,
     :  2,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  701,  710 ) /
     :  0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  0,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0,
     :  1,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  711,  720 ) /
     :  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0, -2,
     :  4,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  4, -5,  0,  0,  0,
     :  2,  0,  0, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  721,  730 ) /
     :  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  1, -1,  0,-18, 17,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1, -1, -1,  0, 20,-20,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  731,  740 ) /
     :  0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  8,-16,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  741,  750 ) /
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  1, -6,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,
     :  1,  0,  0, -1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0,
     :  2, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  751,  760 ) /
     :  2,  0, -1, -1, -1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  761,  770 ) /
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  1, -2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  771,  780 ) /
     :  0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  781,  790 ) /
     :  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  791,  800 ) /
     :  1,  0, -2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0,
     :  2,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  801,  810 ) /
     :  1,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  2,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0,
     :  2, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,
     :  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  3, -1,  0,  0,  0,
     :  1, -1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  811,  820 ) /
     :  3,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -1, -1, -2,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  4,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  821,  830 ) /
     :  3,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  831,  840 ) /
     :  1,  1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -1,  0, -3,  5,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1, -1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  841,  850 ) /
     :  0,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  851,  860 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  861,  870 ) /
     :  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  3, -8,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  871,  880 ) /
     :  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  881,  890 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -5,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  2, -7,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  891,  900 ) /
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -5,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2,
     :  1,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  4,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0,
     :  2,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  901,  910 ) /
     :  1, -1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  911,  920 ) /
     :  2, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -2,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  921,  930 ) /
     :  3,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  931,  940 ) /
     :  0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  0,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  941,  950 ) /
     :  0,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  951,  960 ) /
     :  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0,
     :  2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  961,  NY0 ) /
     :  2,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  1,  2,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^1 terms
      DATA ( ( KY1(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   21,   30 ) /
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   31,   40 ) /
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   41,   50 ) /
     :  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   51,   60 ) /
     :  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1,
     :  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   61,   70 ) /
     :  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   71,   80 ) /
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2,
     :  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   81,   90 ) /
     :  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0,
     :  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   91,  100 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,
     :  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  101,  110 ) /
     :  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  111,  120 ) /
     :  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  121,  130 ) /
     :  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  131,  140 ) /
     :  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2,
     :  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2,
     :  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0,
     :  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  141,  150 ) /
     :  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  151,  160 ) /
     :  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0,
     :  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0,
     :  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  161,  170 ) /
     :  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,
     :  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  0,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  171,  180 ) /
     :  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,
     :  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0,
     :  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,-18, 16,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  181,  190 ) /
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,
     :  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,-10,  3,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0,
     :  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  191,  200 ) /
     :  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2,
     :  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2,
     :  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0,
     :  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  201,  210 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0,
     :  0,  0,  2, -2,  0,  0, -5,  6,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1,
     :  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2,
     :  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  211,  220 ) /
     :  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2,
     :  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0,
     :  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KY1(I,J), I=1,14), J =  221,  230 ) /
     :  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2,
     :  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,
     :  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KY1(I,J), I=1,14), J =  231,  240 ) /
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2,
     :  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,
     :  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0,
     :  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  241,  250 ) /
     :  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2,
     :  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2,
     :  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  251,  260 ) /
     :  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2,
     :  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,
     :  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KY1(I,J), I=1,14), J =  261,  270 ) /
     :  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,
     :  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2,
     :  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2,
     :  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  271,  NY1 ) /
     :  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2,
     :  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2,
     :  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2 /

*  Argument coefficients for t^2 terms
      DATA ( ( KY2(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY2(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY2(I,J), I=1,14), J =   21,  NY2 ) /
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0,
     :  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^3 terms
      DATA ( ( KY3(I,J), I=1,14), J =    1,  NY3 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^4 terms
      DATA ( ( KY4(I,J), I=1,14), J =    1,  NY4 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Sine and cosine coefficients for t^0 terms
      DATA ( ( SY0(I,J), I=1,2), J =    1,   10 ) /
     :            +1538.18D-6,    +9205236.26D-6,
     :             -458.66D-6,     +573033.42D-6,
     :             +137.41D-6,      +97846.69D-6,
     :              -29.05D-6,      -89618.24D-6,
     :              -17.40D-6,      +22438.42D-6,
     :              +31.80D-6,      +20069.50D-6,
     :              +36.70D-6,      +12902.66D-6,
     :              -13.20D-6,       -9592.72D-6,
     :             -192.40D-6,       +7387.02D-6,
     :               +3.92D-6,       -6918.22D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   11,   20 ) /
     :               +0.40D-6,       -5331.13D-6,
     :               -0.90D-6,       -3323.89D-6,
     :               +7.50D-6,       +3143.98D-6,
     :               +7.80D-6,       +2636.13D-6,
     :               -6.60D-6,       +2554.51D-6,
     :               -2.00D-6,       -2423.59D-6,
     :               +6.80D-6,       +1645.01D-6,
     :               +0.00D-6,       -1387.00D-6,
     :               +5.90D-6,       +1323.81D-6,
     :               -0.30D-6,       -1233.89D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   21,   30 ) /
     :               +0.30D-6,       -1075.60D-6,
     :               -4.48D-6,        +852.85D-6,
     :               +0.10D-6,        -800.34D-6,
     :              +35.80D-6,        -674.99D-6,
     :               -1.40D-6,        +695.54D-6,
     :               -0.50D-6,        +684.99D-6,
     :               -2.62D-6,        +643.75D-6,
     :               -1.50D-6,        +522.11D-6,
     :             +273.50D-6,        +164.70D-6,
     :               +1.40D-6,        +335.24D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   31,   40 ) /
     :               +1.90D-6,        +326.60D-6,
     :               +0.40D-6,        +327.11D-6,
     :               -0.50D-6,        -325.03D-6,
     :               -0.40D-6,        +307.03D-6,
     :               +0.50D-6,        +304.17D-6,
     :               -0.10D-6,        -304.46D-6,
     :               -0.40D-6,        -276.81D-6,
     :               +0.90D-6,        +272.05D-6,
     :               +0.30D-6,        +272.22D-6,
     :               +1.20D-6,        +269.45D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   41,   50 ) /
     :               +0.10D-6,        -220.67D-6,
     :             +128.60D-6,         -77.10D-6,
     :               +0.10D-6,        -190.79D-6,
     :             +167.90D-6,          +0.00D-6,
     :               -8.20D-6,        -123.48D-6,
     :               +0.10D-6,        +131.04D-6,
     :               +0.40D-6,        +126.64D-6,
     :               +2.90D-6,        -122.28D-6,
     :               +0.70D-6,        +123.20D-6,
     :               +0.40D-6,        +123.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   51,   60 ) /
     :               -0.30D-6,        +120.70D-6,
     :               -0.50D-6,        +112.90D-6,
     :               -0.20D-6,        -112.94D-6,
     :               +0.20D-6,        +107.31D-6,
     :               -0.30D-6,        -106.20D-6,
     :              +31.90D-6,         -64.10D-6,
     :               +0.00D-6,         +89.50D-6,
     :              +89.10D-6,          +0.00D-6,
     :               +0.00D-6,         +85.32D-6,
     :               -0.20D-6,         -71.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   61,   70 ) /
     :               +0.00D-6,         -70.01D-6,
     :              +13.90D-6,         -55.30D-6,
     :               +0.00D-6,         +67.25D-6,
     :               +0.40D-6,         +66.29D-6,
     :               -0.40D-6,         +64.70D-6,
     :               +1.30D-6,         -60.90D-6,
     :               -0.20D-6,         -60.92D-6,
     :               +0.20D-6,         -59.20D-6,
     :               +1.10D-6,         -55.55D-6,
     :               +0.00D-6,         -55.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   71,   80 ) /
     :               -0.10D-6,         -52.69D-6,
     :               -0.20D-6,         +51.80D-6,
     :               +1.00D-6,         -49.51D-6,
     :               +0.00D-6,         +50.50D-6,
     :               +2.50D-6,         +47.70D-6,
     :               +0.10D-6,         +49.59D-6,
     :               +0.10D-6,         -49.00D-6,
     :              -23.20D-6,         +24.60D-6,
     :               +0.40D-6,         +46.50D-6,
     :               -0.10D-6,         -44.04D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   81,   90 ) /
     :               -0.10D-6,         -42.19D-6,
     :              +13.30D-6,         +26.90D-6,
     :               -0.10D-6,         -39.90D-6,
     :               -0.10D-6,         -39.50D-6,
     :               +0.00D-6,         -39.11D-6,
     :               -0.10D-6,         -38.92D-6,
     :               +0.10D-6,         +36.95D-6,
     :               -0.10D-6,         +34.59D-6,
     :               +0.20D-6,         -32.55D-6,
     :               -0.10D-6,         +31.61D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   91,  100 ) /
     :               +0.00D-6,         +30.40D-6,
     :               +0.20D-6,         +29.40D-6,
     :               +0.00D-6,         -27.91D-6,
     :               +0.10D-6,         +27.50D-6,
     :              -25.70D-6,          -1.70D-6,
     :              +19.90D-6,          +5.90D-6,
     :               +0.00D-6,         +25.80D-6,
     :               +0.20D-6,         +25.20D-6,
     :               +0.00D-6,         -25.31D-6,
     :               +0.20D-6,         +25.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  101,  110 ) /
     :               -0.10D-6,         +24.40D-6,
     :               +0.10D-6,         -24.40D-6,
     :              -23.30D-6,          +0.90D-6,
     :               -0.10D-6,         +24.00D-6,
     :              -18.00D-6,          -5.30D-6,
     :               -0.10D-6,         -22.80D-6,
     :               -0.10D-6,         +22.50D-6,
     :               +0.10D-6,         +21.60D-6,
     :               +0.00D-6,         -21.30D-6,
     :               +0.10D-6,         +20.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  111,  120 ) /
     :               +0.70D-6,         -20.10D-6,
     :               +0.00D-6,         +20.51D-6,
     :              +15.90D-6,          -4.50D-6,
     :               +0.20D-6,         -19.94D-6,
     :               +0.00D-6,         +20.11D-6,
     :              +15.60D-6,          +4.40D-6,
     :               +0.00D-6,         -20.00D-6,
     :               +0.00D-6,         +19.80D-6,
     :               +0.00D-6,         +18.91D-6,
     :               +4.30D-6,         -14.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  121,  130 ) /
     :               -0.10D-6,         -18.50D-6,
     :               -0.10D-6,         +18.40D-6,
     :               +0.00D-6,         +18.10D-6,
     :               +1.00D-6,         +16.81D-6,
     :               -0.10D-6,         -17.60D-6,
     :              -17.60D-6,          +0.00D-6,
     :               -1.30D-6,         -16.26D-6,
     :               +0.00D-6,         -17.41D-6,
     :              +14.50D-6,          -2.70D-6,
     :               +0.00D-6,         +17.08D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  131,  140 ) /
     :               +0.00D-6,         +16.21D-6,
     :               +0.00D-6,         -16.00D-6,
     :               +0.00D-6,         -15.31D-6,
     :               +0.00D-6,         -15.10D-6,
     :               +0.00D-6,         +14.70D-6,
     :               +0.00D-6,         +14.40D-6,
     :               -0.10D-6,         -14.30D-6,
     :               +0.00D-6,         -14.40D-6,
     :               +0.00D-6,         -13.81D-6,
     :               +4.50D-6,          -9.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  141,  150 ) /
     :              -13.80D-6,          +0.00D-6,
     :               +0.00D-6,         -13.38D-6,
     :               -0.10D-6,         +13.10D-6,
     :              +10.30D-6,          +2.70D-6,
     :               +0.00D-6,         +12.80D-6,
     :               +0.00D-6,         -12.80D-6,
     :              +11.70D-6,          +0.80D-6,
     :               +0.00D-6,         -12.00D-6,
     :              +11.30D-6,          +0.50D-6,
     :               +0.00D-6,         +11.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  151,  160 ) /
     :               +0.00D-6,         -11.20D-6,
     :               +0.10D-6,         +10.90D-6,
     :               +0.10D-6,         -10.77D-6,
     :               +0.00D-6,         -10.80D-6,
     :               -0.20D-6,         +10.47D-6,
     :               +0.00D-6,         +10.50D-6,
     :               +0.00D-6,         -10.40D-6,
     :               +0.00D-6,         +10.40D-6,
     :               +0.00D-6,         -10.20D-6,
     :               +0.00D-6,         -10.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  161,  170 ) /
     :               +0.00D-6,          +9.60D-6,
     :               +0.10D-6,          +9.40D-6,
     :               -7.60D-6,          +1.70D-6,
     :               -7.70D-6,          +1.40D-6,
     :               +1.40D-6,          -7.50D-6,
     :               +6.10D-6,          -2.70D-6,
     :               +0.00D-6,          -8.70D-6,
     :               -5.90D-6,          +2.60D-6,
     :               +0.00D-6,          +8.40D-6,
     :               -0.20D-6,          -8.11D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  171,  180 ) /
     :               -2.60D-6,          -5.70D-6,
     :               +0.00D-6,          +8.30D-6,
     :               +2.70D-6,          +5.50D-6,
     :               +4.20D-6,          -4.00D-6,
     :               -0.10D-6,          +8.00D-6,
     :               +0.00D-6,          +8.09D-6,
     :               -1.30D-6,          +6.70D-6,
     :               +0.00D-6,          -7.90D-6,
     :               +0.00D-6,          +7.80D-6,
     :               -7.50D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  181,  190 ) /
     :               -0.50D-6,          -7.20D-6,
     :               +4.90D-6,          +2.70D-6,
     :               +0.00D-6,          +7.50D-6,
     :               +0.00D-6,          -7.50D-6,
     :               +0.00D-6,          -7.49D-6,
     :               +0.00D-6,          -7.20D-6,
     :               +0.10D-6,          +6.90D-6,
     :               -5.60D-6,          +1.40D-6,
     :               -5.70D-6,          -1.30D-6,
     :               +0.00D-6,          +6.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  191,  200 ) /
     :               +4.20D-6,          -2.70D-6,
     :               +0.00D-6,          +6.90D-6,
     :               -3.10D-6,          +3.70D-6,
     :               -3.90D-6,          -2.90D-6,
     :               +0.00D-6,          +6.60D-6,
     :               -3.10D-6,          -3.50D-6,
     :               +1.10D-6,          -5.39D-6,
     :               +0.00D-6,          -6.40D-6,
     :               +0.90D-6,          +5.50D-6,
     :               +0.00D-6,          -6.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  201,  210 ) /
     :               -0.10D-6,          -6.20D-6,
     :               +0.00D-6,          -6.10D-6,
     :               +0.00D-6,          +6.10D-6,
     :               +0.00D-6,          +6.10D-6,
     :               +3.50D-6,          -2.50D-6,
     :               +0.00D-6,          +6.00D-6,
     :               +0.00D-6,          +5.90D-6,
     :               -0.90D-6,          -4.80D-6,
     :               +0.00D-6,          +5.70D-6,
     :               +0.10D-6,          +5.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  211,  220 ) /
     :               +0.00D-6,          +5.70D-6,
     :               +0.00D-6,          +5.70D-6,
     :               +0.00D-6,          +5.60D-6,
     :               +0.00D-6,          +5.60D-6,
     :               +0.20D-6,          -5.40D-6,
     :               -0.90D-6,          -4.70D-6,
     :               -0.40D-6,          -5.10D-6,
     :               +0.00D-6,          +5.50D-6,
     :               +0.00D-6,          -5.40D-6,
     :               +0.00D-6,          -5.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  221,  230 ) /
     :               +1.80D-6,          +3.60D-6,
     :               +0.00D-6,          +5.30D-6,
     :               +0.00D-6,          -5.30D-6,
     :               +0.00D-6,          -5.20D-6,
     :               +0.00D-6,          -5.19D-6,
     :               +3.00D-6,          +2.10D-6,
     :               +0.00D-6,          -5.10D-6,
     :               +0.00D-6,          +5.07D-6,
     :               +0.90D-6,          -4.10D-6,
     :               -5.00D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  231,  240 ) /
     :               +0.00D-6,          -5.00D-6,
     :               +0.00D-6,          +5.00D-6,
     :               +0.00D-6,          -5.00D-6,
     :               +0.00D-6,          -4.90D-6,
     :               +4.90D-6,          +0.00D-6,
     :               +0.00D-6,          -4.90D-6,
     :               +0.90D-6,          +3.90D-6,
     :               +0.00D-6,          +4.80D-6,
     :               -3.70D-6,          -1.10D-6,
     :               +0.00D-6,          -4.72D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  241,  250 ) /
     :               +0.00D-6,          +4.71D-6,
     :               +0.00D-6,          -4.50D-6,
     :               -1.50D-6,          -3.00D-6,
     :               +0.00D-6,          -4.50D-6,
     :               +0.30D-6,          -4.11D-6,
     :               +0.00D-6,          +4.40D-6,
     :               +0.00D-6,          -4.40D-6,
     :               +0.00D-6,          +4.39D-6,
     :               +0.00D-6,          -4.30D-6,
     :               +0.00D-6,          +4.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  251,  260 ) /
     :               +0.00D-6,          -4.30D-6,
     :               +0.20D-6,          +4.03D-6,
     :               +0.20D-6,          +4.00D-6,
     :               -0.60D-6,          +3.50D-6,
     :               +0.00D-6,          +4.10D-6,
     :               +0.00D-6,          +4.00D-6,
     :               +0.00D-6,          -4.00D-6,
     :               +0.00D-6,          -3.91D-6,
     :               +1.90D-6,          +2.00D-6,
     :               +0.00D-6,          +3.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  261,  270 ) /
     :               +0.00D-6,          +3.90D-6,
     :               +0.00D-6,          -3.90D-6,
     :               +3.10D-6,          -0.80D-6,
     :               +0.00D-6,          +3.90D-6,
     :               +0.00D-6,          +3.90D-6,
     :               +0.00D-6,          +3.80D-6,
     :               -0.20D-6,          +3.51D-6,
     :               +0.00D-6,          -3.60D-6,
     :               -2.10D-6,          +1.50D-6,
     :               +0.00D-6,          -3.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  271,  280 ) /
     :               +0.70D-6,          +2.80D-6,
     :               -2.80D-6,          +0.70D-6,
     :               +0.00D-6,          -3.50D-6,
     :               -2.90D-6,          -0.60D-6,
     :               +0.00D-6,          -3.40D-6,
     :               +0.00D-6,          +3.40D-6,
     :               +0.00D-6,          +3.36D-6,
     :               +0.50D-6,          +2.80D-6,
     :               +2.60D-6,          -0.70D-6,
     :               +1.00D-6,          -2.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  281,  290 ) /
     :               +0.00D-6,          -3.30D-6,
     :               +0.00D-6,          +3.30D-6,
     :               +0.00D-6,          +3.23D-6,
     :               +0.00D-6,          +3.20D-6,
     :               +0.00D-6,          -3.20D-6,
     :               +0.00D-6,          -3.20D-6,
     :               +0.00D-6,          +3.20D-6,
     :               +2.90D-6,          -0.30D-6,
     :               +0.08D-6,          +3.05D-6,
     :               -0.70D-6,          -2.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  291,  300 ) /
     :               +0.00D-6,          -3.08D-6,
     :               +0.00D-6,          +3.00D-6,
     :               -1.60D-6,          +1.40D-6,
     :               -2.90D-6,          -0.10D-6,
     :               +0.00D-6,          -2.90D-6,
     :               -2.50D-6,          +0.40D-6,
     :               +0.40D-6,          -2.50D-6,
     :               +0.00D-6,          -2.90D-6,
     :               +0.00D-6,          +2.89D-6,
     :               +0.00D-6,          -2.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  301,  310 ) /
     :               -2.50D-6,          +0.30D-6,
     :               -2.50D-6,          -0.30D-6,
     :               +0.00D-6,          -2.70D-6,
     :               +2.70D-6,          +0.00D-6,
     :               +0.00D-6,          -2.60D-6,
     :               +0.00D-6,          -2.60D-6,
     :               +0.00D-6,          +2.60D-6,
     :               +2.10D-6,          +0.50D-6,
     :               +0.00D-6,          +2.50D-6,
     :               +0.80D-6,          +1.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  311,  320 ) /
     :               +1.90D-6,          -0.60D-6,
     :               +0.00D-6,          -2.50D-6,
     :               +0.00D-6,          -2.40D-6,
     :               +0.00D-6,          +2.40D-6,
     :               +0.00D-6,          -2.40D-6,
     :               +0.00D-6,          +2.40D-6,
     :               -1.90D-6,          +0.50D-6,
     :               -0.10D-6,          -2.30D-6,
     :               +0.00D-6,          +2.30D-6,
     :               +0.00D-6,          -2.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  321,  330 ) /
     :               -1.40D-6,          +0.90D-6,
     :               -0.10D-6,          -2.20D-6,
     :               -0.20D-6,          -2.00D-6,
     :               +0.00D-6,          +2.20D-6,
     :               +0.00D-6,          -2.20D-6,
     :               +0.00D-6,          +2.20D-6,
     :               +0.00D-6,          +2.20D-6,
     :               -1.80D-6,          -0.40D-6,
     :               +0.00D-6,          +2.20D-6,
     :               +0.00D-6,          +2.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  331,  340 ) /
     :               -1.70D-6,          +0.40D-6,
     :               -0.80D-6,          -1.30D-6,
     :               -1.30D-6,          -0.80D-6,
     :               +0.00D-6,          +2.10D-6,
     :               +0.00D-6,          +2.10D-6,
     :               +0.00D-6,          -2.10D-6,
     :               +0.00D-6,          -2.10D-6,
     :               +0.00D-6,          +2.10D-6,
     :               +0.00D-6,          -2.00D-6,
     :               +0.00D-6,          +2.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  341,  350 ) /
     :               +0.00D-6,          +2.00D-6,
     :               +0.00D-6,          +2.00D-6,
     :               +0.00D-6,          -2.00D-6,
     :               +2.00D-6,          +0.00D-6,
     :               +1.10D-6,          -0.90D-6,
     :               +1.60D-6,          -0.40D-6,
     :               +0.00D-6,          -1.91D-6,
     :               +0.00D-6,          -1.90D-6,
     :               +0.00D-6,          +1.90D-6,
     :               +0.00D-6,          -1.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  351,  360 ) /
     :               +0.00D-6,          +1.90D-6,
     :               +1.50D-6,          +0.40D-6,
     :               -1.50D-6,          -0.40D-6,
     :               -1.40D-6,          -0.50D-6,
     :               -1.00D-6,          +0.90D-6,
     :               +0.00D-6,          -1.90D-6,
     :               -0.30D-6,          +1.60D-6,
     :               +0.00D-6,          +1.90D-6,
     :               +0.00D-6,          +1.90D-6,
     :               +0.00D-6,          -1.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  361,  370 ) /
     :               +0.00D-6,          -1.80D-6,
     :               -1.10D-6,          +0.70D-6,
     :               +0.20D-6,          -1.60D-6,
     :               +0.00D-6,          +1.80D-6,
     :               +0.00D-6,          -1.71D-6,
     :               -1.20D-6,          -0.50D-6,
     :               +1.50D-6,          +0.20D-6,
     :               -0.60D-6,          -1.10D-6,
     :               +0.60D-6,          +1.10D-6,
     :               -0.60D-6,          -1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  371,  380 ) /
     :               -1.10D-6,          +0.60D-6,
     :               -1.70D-6,          +0.00D-6,
     :               +0.00D-6,          +1.60D-6,
     :               +0.00D-6,          -1.60D-6,
     :               +0.00D-6,          -1.60D-6,
     :               +1.20D-6,          -0.40D-6,
     :               -0.50D-6,          -1.10D-6,
     :               +0.60D-6,          +1.00D-6,
     :               -1.30D-6,          -0.30D-6,
     :               +0.30D-6,          -1.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  381,  390 ) /
     :               +0.00D-6,          +1.60D-6,
     :               +0.00D-6,          -1.60D-6,
     :               +0.00D-6,          -1.60D-6,
     :               +1.10D-6,          -0.50D-6,
     :               +0.00D-6,          -1.50D-6,
     :               +0.00D-6,          -1.50D-6,
     :               +0.00D-6,          +1.50D-6,
     :               +0.00D-6,          -1.50D-6,
     :               +0.00D-6,          -1.50D-6,
     :               +1.50D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  391,  400 ) /
     :               +0.00D-6,          -1.50D-6,
     :               +1.30D-6,          -0.20D-6,
     :               +0.00D-6,          -1.50D-6,
     :               -1.20D-6,          -0.30D-6,
     :               -1.40D-6,          +0.10D-6,
     :               -0.50D-6,          +1.00D-6,
     :               -0.50D-6,          +1.00D-6,
     :               +0.20D-6,          -1.30D-6,
     :               +0.00D-6,          +1.50D-6,
     :               +0.00D-6,          +1.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  401,  410 ) /
     :               +0.00D-6,          +1.50D-6,
     :               +0.00D-6,          +1.49D-6,
     :               +0.00D-6,          -1.41D-6,
     :               +0.00D-6,          +1.41D-6,
     :               +0.00D-6,          -1.40D-6,
     :               +0.00D-6,          -1.40D-6,
     :               +0.00D-6,          +1.40D-6,
     :               +0.00D-6,          -1.40D-6,
     :               +1.10D-6,          -0.30D-6,
     :               +0.00D-6,          -1.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  411,  420 ) /
     :               +0.00D-6,          +1.40D-6,
     :               +1.40D-6,          +0.00D-6,
     :               -0.30D-6,          +1.10D-6,
     :               +0.20D-6,          +1.20D-6,
     :               -1.30D-6,          +0.00D-6,
     :               +0.00D-6,          -1.30D-6,
     :               +0.00D-6,          +1.30D-6,
     :               -0.70D-6,          -0.60D-6,
     :               -0.80D-6,          +0.50D-6,
     :               -0.20D-6,          -1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  421,  430 ) /

     :               +1.10D-6,          +0.20D-6,
     :               +0.00D-6,          -1.30D-6,
     :               +0.00D-6,          -1.30D-6,
     :               +0.00D-6,          -1.30D-6,
     :               +0.00D-6,          -1.30D-6,
     :               +0.00D-6,          -1.29D-6,
     :               +0.00D-6,          +1.20D-6,
     :               +0.00D-6,          -1.20D-6,
     :               -0.40D-6,          -0.80D-6,
     :               +0.00D-6,          +1.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  431,  440 ) /
     :               +1.20D-6,          +0.00D-6,
     :               -0.70D-6,          -0.50D-6,
     :               -1.00D-6,          +0.20D-6,
     :               -1.00D-6,          +0.20D-6,
     :               +0.20D-6,          -1.00D-6,
     :               +0.40D-6,          +0.80D-6,
     :               -0.40D-6,          +0.80D-6,
     :               +0.00D-6,          -1.20D-6,
     :               +0.00D-6,          +1.15D-6,
     :               +0.00D-6,          +1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  441,  450 ) /
     :               -0.20D-6,          +0.90D-6,
     :               -1.10D-6,          +0.00D-6,
     :               +0.00D-6,          -1.10D-6,
     :               -1.10D-6,          +0.00D-6,
     :               +0.00D-6,          +1.10D-6,
     :               +0.00D-6,          +1.10D-6,
     :               +0.00D-6,          +1.10D-6,
     :               +0.60D-6,          -0.50D-6,
     :               -0.90D-6,          -0.20D-6,
     :               -0.40D-6,          -0.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  451,  460 ) /
     :               -0.50D-6,          +0.60D-6,
     :               +0.00D-6,          +1.10D-6,
     :               +0.00D-6,          -1.10D-6,
     :               +0.00D-6,          +1.00D-6,
     :               +1.00D-6,          +0.00D-6,
     :               +0.80D-6,          -0.20D-6,
     :               +0.00D-6,          +1.00D-6,
     :               +0.00D-6,          +1.00D-6,
     :               +0.00D-6,          -1.00D-6,
     :               -1.00D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  461,  470 ) /
     :               +0.00D-6,          +1.00D-6,
     :               +1.00D-6,          +0.00D-6,
     :               +1.00D-6,          +0.00D-6,
     :               -0.80D-6,          -0.20D-6,
     :               +0.40D-6,          +0.60D-6,
     :               -0.40D-6,          -0.60D-6,
     :               +0.00D-6,          -1.00D-6,
     :               +0.00D-6,          +1.00D-6,
     :               +0.00D-6,          +1.00D-6,
     :               +0.00D-6,          +1.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  471,  480 ) /
     :               +0.00D-6,          +1.00D-6,
     :               +0.00D-6,          -1.00D-6,
     :               +0.00D-6,          +0.91D-6,
     :               +0.10D-6,          +0.80D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          -0.90D-6,
     :               +0.00D-6,          -0.90D-6,
     :               -0.70D-6,          -0.20D-6,
     :               +0.70D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  481,  490 ) /
     :               -0.30D-6,          +0.60D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          -0.90D-6,
     :               -0.50D-6,          -0.40D-6,
     :               -0.90D-6,          +0.00D-6,
     :               +0.00D-6,          -0.90D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          +0.90D-6,
     :               +0.00D-6,          -0.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  491,  500 ) /
     :               +0.00D-6,          -0.90D-6,
     :               +0.00D-6,          -0.80D-6,
     :               +0.00D-6,          +0.80D-6,
     :               +0.00D-6,          -0.80D-6,
     :               +0.10D-6,          +0.70D-6,
     :               -0.70D-6,          +0.10D-6,
     :               -0.60D-6,          +0.20D-6,
     :               +0.20D-6,          +0.60D-6,
     :               +0.00D-6,          +0.80D-6,
     :               -0.50D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  501,  510 ) /
     :               -0.50D-6,          -0.30D-6,
     :               -0.50D-6,          -0.30D-6,
     :               +0.00D-6,          -0.80D-6,
     :               -0.30D-6,          +0.50D-6,
     :               -0.80D-6,          +0.00D-6,
     :               -0.30D-6,          -0.50D-6,
     :               -0.30D-6,          +0.50D-6,
     :               -0.30D-6,          -0.50D-6,
     :               +0.00D-6,          +0.80D-6,
     :               +0.00D-6,          -0.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  511,  520 ) /
     :               +0.00D-6,          -0.80D-6,
     :               +0.00D-6,          -0.80D-6,
     :               +0.00D-6,          +0.80D-6,
     :               +0.00D-6,          +0.80D-6,
     :               +0.00D-6,          -0.80D-6,
     :               +0.00D-6,          +0.76D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.10D-6,          -0.60D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.70D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  521,  530 ) /
     :               +0.00D-6,          -0.70D-6,
     :               +0.00D-6,          -0.70D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.00D-6,          -0.70D-6,
     :               -0.70D-6,          +0.00D-6,
     :               -0.50D-6,          +0.20D-6,
     :               -0.20D-6,          -0.50D-6,
     :               +0.50D-6,          -0.20D-6,
     :               +0.20D-6,          +0.50D-6,
     :               -0.20D-6,          -0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  531,  540 ) /
     :               +0.50D-6,          -0.20D-6,
     :               -0.50D-6,          +0.20D-6,
     :               +0.00D-6,          -0.70D-6,
     :               +0.00D-6,          -0.70D-6,
     :               +0.70D-6,          +0.00D-6,
     :               -0.60D-6,          -0.10D-6,
     :               +0.60D-6,          -0.10D-6,
     :               +0.40D-6,          +0.30D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.70D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  541,  550 ) /
     :               +0.00D-6,          +0.70D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.00D-6,          +0.70D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.10D-6,          -0.50D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.40D-6,          +0.20D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  551,  560 ) /
     :               +0.00D-6,          +0.60D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.50D-6,          +0.10D-6,
     :               -0.50D-6,          -0.10D-6,
     :               -0.10D-6,          -0.50D-6,
     :               +0.10D-6,          +0.50D-6,
     :               +0.50D-6,          -0.10D-6,
     :               -0.10D-6,          +0.50D-6,
     :               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  561,  570 ) /
     :               -0.40D-6,          +0.20D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.60D-6,          +0.00D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.20D-6,          +0.40D-6,
     :               -0.40D-6,          +0.20D-6,
     :               +0.30D-6,          +0.30D-6,
     :               +0.40D-6,          -0.20D-6,
     :               -0.40D-6,          -0.20D-6,
     :               +0.00D-6,          +0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  571,  580 ) /
     :               +0.00D-6,          +0.60D-6,
     :               +0.40D-6,          +0.20D-6,
     :               -0.20D-6,          -0.40D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.00D-6,          +0.60D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.00D-6,          -0.60D-6,
     :               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  581,  590 ) /
     :               +0.00D-6,          -0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.10D-6,          +0.40D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.10D-6,          +0.40D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.30D-6,          -0.20D-6,
     :               -0.20D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  591,  600 ) /
     :               +0.20D-6,          +0.30D-6,
     :               +0.40D-6,          -0.10D-6,
     :               +0.40D-6,          +0.10D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.30D-6,          +0.20D-6,
     :               -0.30D-6,          +0.20D-6,
     :               +0.20D-6,          +0.30D-6,
     :               -0.30D-6,          +0.20D-6,
     :               +0.00D-6,          +0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  601,  610 ) /
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               -0.50D-6,          +0.00D-6,
     :               +0.50D-6,          +0.00D-6,
     :               +0.00D-6,          -0.50D-6,
     :               -0.50D-6,          +0.00D-6,
     :               -0.50D-6,          +0.00D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.40D-6,          +0.10D-6,
     :               -0.40D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  611,  620 ) /
     :               +0.40D-6,          -0.10D-6,
     :               -0.40D-6,          +0.10D-6,
     :               +0.10D-6,          +0.40D-6,
     :               +0.10D-6,          +0.40D-6,
     :               -0.50D-6,          +0.00D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          -0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  621,  630 ) /
     :               +0.50D-6,          +0.00D-6,
     :               +0.00D-6,          -0.50D-6,
     :               +0.00D-6,          +0.50D-6,
     :               +0.00D-6,          -0.40D-6,
     :               -0.20D-6,          +0.20D-6,
     :               -0.10D-6,          +0.30D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  631,  640 ) /
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               -0.10D-6,          +0.30D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               -0.20D-6,          -0.20D-6,
     :               +0.20D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  641,  650 ) /
     :               +0.20D-6,          +0.20D-6,
     :               -0.10D-6,          +0.30D-6,
     :               -0.30D-6,          +0.10D-6,
     :               +0.10D-6,          +0.30D-6,
     :               -0.10D-6,          +0.30D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  651,  660 ) /
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               -0.40D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  661,  670 ) /
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6,
     :               -0.40D-6,          +0.00D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.40D-6,          +0.00D-6,
     :               +0.00D-6,          -0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  671,  680 ) /
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               -0.10D-6,          +0.30D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          +0.40D-6,
     :               +0.00D-6,          -0.40D-6,
     :               +0.40D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  681,  690 ) /
     :               +0.00D-6,          +0.40D-6,
     :               +0.21D-6,          +0.10D-6,
     :               +0.00D-6,          +0.30D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.20D-6,          +0.10D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  691,  700 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               -0.20D-6,          +0.10D-6,
     :               -0.10D-6,          -0.20D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.10D-6,          -0.20D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  701,  710 ) /
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               -0.20D-6,          +0.10D-6,
     :               +0.00D-6,          -0.30D-6,
     :               -0.10D-6,          -0.20D-6,
     :               -0.10D-6,          +0.20D-6,
     :               +0.20D-6,          -0.10D-6,
     :               -0.10D-6,          -0.20D-6,
     :               +0.20D-6,          +0.10D-6,
     :               +0.20D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  711,  720 ) /
     :               -0.20D-6,          -0.10D-6,
     :               -0.10D-6,          -0.20D-6,
     :               +0.20D-6,          -0.10D-6,
     :               +0.20D-6,          +0.10D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.30D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  721,  730 ) /
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  731,  740 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.30D-6,          +0.00D-6,
     :               -0.30D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  741,  750 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  751,  760 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  761,  770 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          +0.30D-6,
     :               -0.30D-6,          +0.00D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.00D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  771,  780 ) /
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               -0.20D-6,          -0.10D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.30D-6,
     :               +0.00D-6,          -0.21D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  781,  790 ) /
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  791,  800 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  801,  810 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.10D-6,          -0.10D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  811,  820 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  821,  830 ) /
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  831,  840 ) /
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  841,  850 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  851,  860 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  861,  870 ) /
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  871,  880 ) /
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  881,  890 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  891,  900 ) /
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.10D-6,          -0.10D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  901,  910 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.20D-6,          +0.00D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  911,  920 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  921,  930 ) /
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               -0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  931,  940 ) /
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          +0.20D-6,
     :               +0.00D-6,          -0.19D-6,
     :               +0.00D-6,          +0.17D-6,
     :               +0.00D-6,          +0.11D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.10D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  941,  950 ) /
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               -0.10D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  951,  960 ) /
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.10D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          -0.10D-6,
     :               +0.00D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  961,  NY0 ) /
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          +0.10D-6 /

*  Sine and cosine coefficients for t^1 terms
      DATA ( ( SY1(I,J), I=1,2), J =    1,   10 ) /
     :          +153041.82D-6,        +878.89D-6,
     :           +11714.49D-6,        -289.32D-6,
     :            +2024.68D-6,         -50.99D-6,
     :            -1837.33D-6,         +47.75D-6,
     :            -1312.21D-6,         -28.91D-6,
     :             -632.54D-6,          +0.78D-6,
     :             +459.68D-6,         -67.23D-6,
     :             +344.50D-6,          +1.46D-6,
     :             +268.14D-6,          -7.03D-6,
     :             +192.06D-6,         +29.80D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   11,   20 ) /
     :             +139.64D-6,          +0.15D-6,
     :             -113.94D-6,          -1.06D-6,
     :             +109.81D-6,          +3.18D-6,
     :              -56.37D-6,          +0.13D-6,
     :              -56.17D-6,          -0.02D-6,
     :              -53.05D-6,          -1.23D-6,
     :              -51.60D-6,          +0.17D-6,
     :              +45.91D-6,          -0.11D-6,
     :              -42.45D-6,          +0.02D-6,
     :              +40.82D-6,          -1.03D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   21,   30 ) /
     :              +34.30D-6,          -1.24D-6,
     :              +28.89D-6,          +0.00D-6,
     :              +27.61D-6,          -1.22D-6,
     :              -25.43D-6,          +1.00D-6,
     :              -26.01D-6,          +0.07D-6,
     :              -23.02D-6,          +0.06D-6,
     :              +19.37D-6,          -0.01D-6,
     :              +14.05D-6,          -4.19D-6,
     :              +18.18D-6,          -0.01D-6,
     :              -14.86D-6,          -0.09D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   31,   40 ) /
     :              +13.49D-6,          -0.01D-6,
     :              +12.44D-6,          -0.27D-6,
     :              +11.46D-6,          +0.03D-6,
     :              -11.33D-6,          -0.06D-6,
     :               -9.81D-6,          +0.01D-6,
     :               -9.08D-6,          -0.02D-6,
     :               +2.74D-6,          -4.56D-6,
     :               +6.84D-6,          -0.04D-6,
     :               -6.73D-6,          +0.01D-6,
     :               +6.54D-6,          +0.01D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   41,   50 ) /
     :               -6.35D-6,          -0.01D-6,
     :               +5.90D-6,          -0.02D-6,
     :               -5.85D-6,          +0.02D-6,
     :               -5.73D-6,          +0.01D-6,
     :               +5.60D-6,          +0.00D-6,
     :               -5.16D-6,          +0.00D-6,
     :               -5.14D-6,          +0.01D-6,
     :               +4.76D-6,          -0.02D-6,
     :               -4.40D-6,          +0.02D-6,
     :               -4.22D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   51,   60 ) /
     :               -4.20D-6,          +0.01D-6,
     :               +3.58D-6,          +0.31D-6,
     :               +3.87D-6,          +0.01D-6,
     :               +3.76D-6,          +0.00D-6,
     :               -3.62D-6,          -0.01D-6,
     :               -3.61D-6,          +0.00D-6,
     :               -1.28D-6,          -2.14D-6,
     :               -3.18D-6,          +0.00D-6,
     :               +3.01D-6,          +0.00D-6,
     :               -2.97D-6,          +0.01D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   61,   70 ) /
     :               +2.91D-6,          +0.00D-6,
     :               -2.73D-6,          +0.00D-6,
     :               +2.58D-6,          -0.01D-6,
     :               +2.56D-6,          -0.01D-6,
     :               -2.51D-6,          -0.01D-6,
     :               -2.35D-6,          -0.01D-6,
     :               -2.21D-6,          +0.01D-6,
     :               -2.04D-6,          +0.01D-6,
     :               -1.94D-6,          +0.00D-6,
     :               +0.41D-6,          -1.43D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   71,   80 ) /
     :               -1.84D-6,          +0.00D-6,
     :               -1.77D-6,          +0.01D-6,
     :               +0.00D-6,          +1.77D-6,
     :               +1.76D-6,          +0.00D-6,
     :               -1.07D-6,          -0.53D-6,
     :               -1.48D-6,          +0.00D-6,
     :               -1.40D-6,          +0.01D-6,
     :               -1.35D-6,          -0.01D-6,
     :               -1.32D-6,          +0.00D-6,
     :               -1.28D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   81,   90 ) /
     :               +1.24D-6,          +0.00D-6,
     :               +1.23D-6,          +0.00D-6,
     :               +1.19D-6,          +0.00D-6,
     :               +1.18D-6,          -0.01D-6,
     :               +1.17D-6,          +0.00D-6,
     :               -1.15D-6,          +0.00D-6,
     :               +1.14D-6,          +0.00D-6,
     :               -1.14D-6,          +0.00D-6,
     :               +1.09D-6,          +0.03D-6,
     :               -1.08D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   91,  100 ) /
     :               +1.04D-6,          +0.00D-6,
     :               +1.02D-6,          +0.00D-6,
     :               +0.98D-6,          -0.01D-6,
     :               +0.91D-6,          +0.02D-6,
     :               +0.00D-6,          +0.93D-6,
     :               -0.91D-6,          +0.00D-6,
     :               -0.90D-6,          +0.00D-6,
     :               +0.86D-6,          +0.00D-6,
     :               -0.84D-6,          +0.00D-6,
     :               -0.83D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  101,  110 ) /
     :               -0.82D-6,          +0.00D-6,
     :               +0.41D-6,          +0.39D-6,
     :               +0.40D-6,          -0.38D-6,
     :               +0.78D-6,          +0.00D-6,
     :               +0.74D-6,          +0.00D-6,
     :               -0.73D-6,          +0.00D-6,
     :               +0.68D-6,          +0.00D-6,
     :               +0.66D-6,          +0.00D-6,
     :               -0.64D-6,          +0.00D-6,
     :               -0.63D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  111,  120 ) /
     :               +0.63D-6,          +0.00D-6,
     :               +0.62D-6,          +0.00D-6,
     :               +0.60D-6,          +0.00D-6,
     :               -0.59D-6,          +0.00D-6,
     :               -0.59D-6,          +0.00D-6,
     :               +0.59D-6,          +0.00D-6,
     :               +0.57D-6,          +0.00D-6,
     :               +0.38D-6,          -0.19D-6,
     :               -0.01D-6,          -0.55D-6,
     :               +0.44D-6,          -0.11D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  121,  130 ) /
     :               +0.53D-6,          +0.00D-6,
     :               -0.53D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               -0.52D-6,          +0.00D-6,
     :               +0.53D-6,          +0.00D-6,
     :               +0.52D-6,          +0.00D-6,
     :               +0.51D-6,          +0.00D-6,
     :               +0.51D-6,          +0.00D-6,
     :               -0.21D-6,          -0.30D-6,
     :               -0.50D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  131,  140 ) /
     :               -0.11D-6,          +0.37D-6,
     :               -0.11D-6,          +0.37D-6,
     :               -0.48D-6,          +0.00D-6,
     :               -0.46D-6,          -0.01D-6,
     :               -0.47D-6,          +0.00D-6,
     :               -0.03D-6,          +0.43D-6,
     :               +0.45D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6,
     :               -0.44D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  141,  150 ) /
     :               +0.43D-6,          +0.00D-6,
     :               +0.44D-6,          +0.00D-6,
     :               +0.42D-6,          +0.00D-6,
     :               -0.42D-6,          +0.00D-6,
     :               +0.41D-6,          +0.00D-6,
     :               -0.41D-6,          +0.00D-6,
     :               +0.02D-6,          +0.39D-6,
     :               +0.40D-6,          +0.00D-6,
     :               -0.40D-6,          +0.00D-6,
     :               -0.39D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  151,  160 ) /
     :               +0.39D-6,          +0.00D-6,
     :               +0.15D-6,          -0.24D-6,
     :               -0.37D-6,          -0.01D-6,
     :               +0.37D-6,          +0.00D-6,
     :               -0.37D-6,          +0.00D-6,
     :               -0.37D-6,          +0.00D-6,
     :               -0.31D-6,          +0.06D-6,
     :               -0.35D-6,          +0.00D-6,
     :               +0.35D-6,          +0.00D-6,
     :               -0.07D-6,          -0.27D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  161,  170 ) /
     :               -0.33D-6,          +0.01D-6,
     :               -0.33D-6,          +0.00D-6,
     :               +0.07D-6,          -0.26D-6,
     :               +0.33D-6,          +0.00D-6,
     :               +0.00D-6,          -0.32D-6,
     :               +0.32D-6,          +0.00D-6,
     :               -0.32D-6,          +0.00D-6,
     :               +0.32D-6,          +0.00D-6,
     :               -0.24D-6,          -0.07D-6,
     :               +0.24D-6,          +0.07D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  171,  180 ) /
     :               +0.30D-6,          +0.00D-6,
     :               +0.08D-6,          -0.22D-6,
     :               -0.30D-6,          +0.00D-6,
     :               -0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.30D-6,          +0.00D-6,
     :               +0.00D-6,          -0.29D-6,
     :               +0.00D-6,          -0.29D-6,
     :               +0.20D-6,          -0.09D-6,
     :               +0.29D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  181,  190 ) /
     :               -0.05D-6,          -0.24D-6,
     :               +0.29D-6,          +0.00D-6,
     :               -0.27D-6,          +0.00D-6,
     :               -0.19D-6,          -0.08D-6,
     :               -0.27D-6,          +0.00D-6,
     :               +0.25D-6,          +0.00D-6,
     :               +0.25D-6,          +0.00D-6,
     :               -0.25D-6,          +0.00D-6,
     :               +0.25D-6,          +0.00D-6,
     :               -0.25D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  191,  200 ) /
     :               -0.01D-6,          +0.23D-6,
     :               -0.23D-6,          +0.00D-6,
     :               +0.23D-6,          +0.00D-6,
     :               +0.23D-6,          +0.00D-6,
     :               -0.15D-6,          -0.07D-6,
     :               -0.23D-6,          +0.00D-6,
     :               -0.22D-6,          +0.00D-6,
     :               +0.22D-6,          +0.00D-6,
     :               -0.22D-6,          +0.00D-6,
     :               -0.22D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  201,  210 ) /
     :               +0.04D-6,          -0.17D-6,
     :               -0.01D-6,          -0.21D-6,
     :               +0.08D-6,          -0.14D-6,
     :               -0.01D-6,          +0.19D-6,
     :               +0.21D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.20D-6,          +0.00D-6,
     :               -0.04D-6,          -0.16D-6,
     :               +0.19D-6,          +0.00D-6,
     :               +0.19D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  211,  220 ) /
     :               -0.19D-6,          +0.00D-6,
     :               +0.18D-6,          +0.00D-6,
     :               -0.18D-6,          +0.00D-6,
     :               +0.18D-6,          +0.00D-6,
     :               +0.17D-6,          +0.00D-6,
     :               -0.12D-6,          +0.06D-6,
     :               +0.13D-6,          -0.04D-6,
     :               -0.11D-6,          +0.06D-6,
     :               +0.17D-6,          +0.00D-6,
     :               +0.16D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  221,  230 ) /
     :               -0.17D-6,          +0.00D-6,
     :               -0.17D-6,          +0.00D-6,
     :               -0.14D-6,          +0.02D-6,
     :               +0.14D-6,          +0.03D-6,
     :               +0.00D-6,          +0.15D-6,
     :               -0.15D-6,          +0.00D-6,
     :               -0.14D-6,          +0.01D-6,
     :               +0.16D-6,          +0.00D-6,
     :               -0.06D-6,          +0.10D-6,
     :               +0.05D-6,          +0.10D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  231,  240 ) /
     :               +0.02D-6,          +0.13D-6,
     :               -0.11D-6,          +0.04D-6,
     :               -0.12D-6,          -0.02D-6,
     :               -0.05D-6,          -0.10D-6,
     :               +0.14D-6,          +0.00D-6,
     :               -0.09D-6,          +0.05D-6,
     :               +0.00D-6,          +0.14D-6,
     :               +0.14D-6,          +0.00D-6,
     :               -0.14D-6,          +0.00D-6,
     :               +0.04D-6,          +0.10D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  241,  250 ) /
     :               -0.06D-6,          +0.08D-6,
     :               +0.05D-6,          +0.09D-6,
     :               -0.14D-6,          +0.00D-6,
     :               +0.08D-6,          +0.06D-6,
     :               +0.14D-6,          +0.00D-6,
     :               +0.14D-6,          +0.00D-6,
     :               +0.13D-6,          +0.00D-6,
     :               -0.07D-6,          +0.06D-6,
     :               +0.11D-6,          -0.02D-6,
     :               -0.13D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  251,  260 ) /
     :               -0.13D-6,          +0.00D-6,
     :               -0.13D-6,          +0.00D-6,
     :               -0.13D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               +0.12D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               +0.00D-6,          -0.12D-6,
     :               -0.02D-6,          -0.09D-6,
     :               +0.02D-6,          -0.09D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  261,  270 ) /
     :               -0.11D-6,          +0.00D-6,
     :               +0.11D-6,          +0.00D-6,
     :               +0.07D-6,          -0.04D-6,
     :               +0.11D-6,          +0.00D-6,
     :               +0.11D-6,          +0.00D-6,
     :               -0.11D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6,
     :               -0.10D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  271,  NY1 ) /
     :               +0.10D-6,          +0.00D-6,
     :               +0.00D-6,          +0.10D-6,
     :               +0.00D-6,          +0.10D-6,
     :               -0.10D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6,
     :               -0.10D-6,          +0.00D-6,
     :               +0.10D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^2 terms
      DATA ( ( SY2(I,J), I=1,2), J =    1,   10 ) /
     :             +121.15D-6,       -2301.27D-6,
     :               -0.98D-6,        -143.27D-6,
     :               -0.27D-6,         -24.46D-6,
     :               +0.24D-6,         +22.41D-6,
     :               -1.19D-6,          -5.61D-6,
     :               +3.57D-6,          -1.83D-6,
     :               +0.24D-6,          -5.02D-6,
     :               -0.04D-6,          -3.23D-6,
     :               -0.48D-6,          +2.40D-6,
     :               -0.10D-6,          +1.73D-6 /
      DATA ( ( SY2(I,J), I=1,2), J =   11,   20 ) /
     :               -0.01D-6,          +1.33D-6,
     :               -0.04D-6,          +0.83D-6,
     :               -0.05D-6,          -0.79D-6,
     :               +0.03D-6,          -0.66D-6,
     :               +0.00D-6,          -0.64D-6,
     :               +0.04D-6,          +0.61D-6,
     :               -0.01D-6,          -0.41D-6,
     :               -0.01D-6,          +0.35D-6,
     :               -0.01D-6,          -0.33D-6,
     :               +0.01D-6,          +0.31D-6 /
      DATA ( ( SY2(I,J), I=1,2), J =   21,  NY2 ) /
     :               +0.01D-6,          +0.27D-6,
     :               -0.07D-6,          -0.17D-6,
     :               +0.07D-6,          +0.17D-6,
     :               +0.02D-6,          -0.21D-6,
     :               +0.01D-6,          +0.20D-6,
     :               +0.01D-6,          -0.17D-6,
     :               +0.01D-6,          -0.16D-6,
     :               +0.00D-6,          -0.13D-6,
     :               -0.07D-6,          -0.04D-6,
     :               +0.02D-6,          +0.08D-6 /

*  Sine and cosine coefficients for t^3 terms
      DATA ( ( SY3(I,J), I=1,2), J =    1,  NY3 ) /
     :              -15.23D-6,          -1.62D-6,
     :               -1.16D-6,          -0.01D-6,
     :               -0.20D-6,          +0.00D-6,
     :               +0.18D-6,          +0.00D-6,
     :               +0.13D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^4 terms
      DATA ( ( SY4(I,J), I=1,2), J =    1,  NY4 ) /
     :               -0.01D-6,          +0.11D-6 /

*  ---------------------------------------
*  The series for s+XY/2: numerical values
*  ---------------------------------------

*  Polynomial coefficients
      DATA SP /    94    D-6,
     :           3808.35 D-6,
     :           -119.94 D-6,
     :         -72574.09 D-6,
     :             27.70 D-6,
     :             15.61 D-6 /

*  Argument coefficients for t^0
      DATA ( ( KS0(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   21,   30 ) /
     :  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  4, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   31,  NS0 ) /
     :  0,  0,  2, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^1
      DATA ( ( KS1(I,J), I=1,14), J =    1,  NS1 ) /
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^2
      DATA ( ( KS2(I,J), I=1,14), J =    1,   10 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS2(I,J), I=1,14), J =   11,   20 ) /
     :  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS2(I,J), I=1,14), J =   21,  NS2 ) /
     :  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^3
      DATA ( ( KS3(I,J), I=1,14), J =    1,  NS3 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     :  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Argument coefficients for t^4
      DATA ( ( KS4(I,J), I=1,14), J =    1,  NS4 ) /
     :  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /

*  Sine and cosine coefficients for t^0
      DATA ( ( SS0(I,J), I=1,2), J =    1,   10 ) /
     :            -2640.73D-6,          +0.39D-6,
     :              -63.53D-6,          +0.02D-6,
     :              -11.75D-6,          -0.01D-6,
     :              -11.21D-6,          -0.01D-6,
     :               +4.57D-6,          +0.00D-6,
     :               -2.02D-6,          +0.00D-6,
     :               -1.98D-6,          +0.00D-6,
     :               +1.72D-6,          +0.00D-6,
     :               +1.41D-6,          +0.01D-6,
     :               +1.26D-6,          +0.01D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   11,   20 ) /
     :               +0.63D-6,          +0.00D-6,
     :               +0.63D-6,          +0.00D-6,
     :               -0.46D-6,          +0.00D-6,
     :               -0.45D-6,          +0.00D-6,
     :               -0.36D-6,          +0.00D-6,
     :               +0.24D-6,          +0.12D-6,
     :               -0.32D-6,          +0.00D-6,
     :               -0.28D-6,          +0.00D-6,
     :               -0.27D-6,          +0.00D-6,
     :               -0.26D-6,          +0.00D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   21,   30 ) /
     :               +0.21D-6,          +0.00D-6,
     :               -0.19D-6,          +0.00D-6,
     :               -0.18D-6,          +0.00D-6,
     :               +0.10D-6,          -0.05D-6,
     :               -0.15D-6,          +0.00D-6,
     :               +0.14D-6,          +0.00D-6,
     :               +0.14D-6,          +0.00D-6,
     :               -0.14D-6,          +0.00D-6,
     :               -0.14D-6,          +0.00D-6,
     :               -0.13D-6,          +0.00D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   31,  NS0 ) /
     :               +0.11D-6,          +0.00D-6,
     :               -0.11D-6,          +0.00D-6,
     :               -0.11D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^1
      DATA ( ( SS1(I,J), I=1,2), J =    1,  NS1 ) /
     :               -0.07D-6,          +3.57D-6,
     :               +1.71D-6,          -0.03D-6,
     :               +0.00D-6,          +0.48D-6 /

*  Sine and cosine coefficients for t^2
      DATA ( ( SS2(I,J), I=1,2), J =    1,   10 ) /
     :             +743.53D-6,          -0.17D-6,
     :              +56.91D-6,          +0.06D-6,
     :               +9.84D-6,          -0.01D-6,
     :               -8.85D-6,          +0.01D-6,
     :               -6.38D-6,          -0.05D-6,
     :               -3.07D-6,          +0.00D-6,
     :               +2.23D-6,          +0.00D-6,
     :               +1.67D-6,          +0.00D-6,
     :               +1.30D-6,          +0.00D-6,
     :               +0.93D-6,          +0.00D-6 /
      DATA ( ( SS2(I,J), I=1,2), J =   11,   20 ) /
     :               +0.68D-6,          +0.00D-6,
     :               -0.55D-6,          +0.00D-6,
     :               +0.53D-6,          +0.00D-6,
     :               -0.27D-6,          +0.00D-6,
     :               -0.27D-6,          +0.00D-6,
     :               -0.26D-6,          +0.00D-6,
     :               -0.25D-6,          +0.00D-6,
     :               +0.22D-6,          +0.00D-6,
     :               -0.21D-6,          +0.00D-6,
     :               +0.20D-6,          +0.00D-6 /
      DATA ( ( SS2(I,J), I=1,2), J =   21,  NS2 ) /
     :               +0.17D-6,          +0.00D-6,
     :               +0.13D-6,          +0.00D-6,
     :               -0.13D-6,          +0.00D-6,
     :               -0.12D-6,          +0.00D-6,
     :               -0.11D-6,          +0.00D-6 /

*  Sine and cosine coefficients for t^3
      DATA ( ( SS3(I,J), I=1,2), J =    1,  NS3 ) /
     :               +0.30D-6,         -23.51D-6,
     :               -0.03D-6,          -1.39D-6,
     :               -0.01D-6,          -0.24D-6,
     :               +0.00D-6,          +0.22D-6 /

*  Sine and cosine coefficients for t^4
      DATA ( ( SS4(I,J), I=1,2), J =    1,  NS4 ) /
     :               -0.26D-6,          -0.01D-6 /

* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

*  Interval between fundamental epoch J2000.0 and current date (JC).
      T = ( ( DATE1-DJ0 ) + DATE2 ) / DJC
 
*  Fundamental Arguments

*  Mean Anomaly of the Moon.
      FA(1) = iau_ANPM ( ( 485868.249036D0 +
     :                   ( 715923.2178D0 +
     :                   (     31.8792D0 +
     :                   (      0.051635D0 +
     :                   (     -0.00024470D0 )
     :                   * T ) * T ) * T ) * T ) * DAS2R
     :                   + MOD ( 1325D0*T, 1D0 ) * D2PI )

*  Mean Anomaly of the Sun.
      FA(2) = iau_ANPM ( ( 1287104.793048D0 +
     :                   ( 1292581.0481D0 +
     :                   (      -0.5532D0 +
     :                   (      +0.000136D0 +
     :                   (      -0.00001149D0 )
     :                   * T ) * T ) * T ) * T ) * DAS2R
     :                   + MOD ( 99D0*T, 1D0 ) * D2PI )

*  Mean Longitude of the Moon minus Mean Longitude of the Ascending
*  Node of the Moon.
      FA(3) = iau_ANPM ( (  335779.526232D0 +
     :                   (  295262.8478D0 +
     :                   (     -12.7512D0 +
     :                   (      -0.001037D0 +
     :                   (       0.00000417D0 )
     :                   * T ) * T ) * T ) * T ) * DAS2R
     :                   + MOD ( 1342D0*T, 1D0 ) * D2PI )

*  Mean Elongation of the Moon from the Sun.
      FA(4) = iau_ANPM ( ( 1072260.703692D0 +
     :                   ( 1105601.2090D0 +
     :                   (      -6.3706D0 +
     :                   (       0.006593D0 +
     :                   (      -0.00003169D0 )
     :                   * T ) * T ) * T ) * T ) * DAS2R
     :                   + MOD ( 1236D0*T, 1D0 ) * D2PI )

*  Mean Longitude of the Ascending Node of the Moon.
      FA(5) = iau_ANPM ( (  450160.398036D0 +
     :                   ( -482890.5431D0 +
     :                   (       7.4722D0 +
     :                   (       0.007702D0 +
     :                   (      -0.00005939D0 )
     :                   * T ) * T ) * T ) * T ) * DAS2R
     :                   + MOD ( -5D0*T, 1D0 ) * D2PI )

      FA( 6) = iau_ANPM ( 4.402608842D0 + 2608.7903141574D0 * T )
      FA( 7) = iau_ANPM ( 3.176146697D0 + 1021.3285546211D0 * T )
      FA( 8) = iau_ANPM ( 1.753470314D0 +  628.3075849991D0 * T )
      FA( 9) = iau_ANPM ( 6.203480913D0 +  334.0612426700D0 * T )
      FA(10) = iau_ANPM ( 0.599546497D0 +   52.9690962641D0 * T )
      FA(11) = iau_ANPM ( 0.874016757D0 +   21.3299104960D0 * T )
      FA(12) = iau_ANPM ( 5.481293872D0 +    7.4781598567D0 * T )
      FA(13) = iau_ANPM ( 5.311886287D0 +    3.8133035638D0 * T )
      FA(14) =          ( 0.024381750D0 +    0.00000538691D0 * T ) * T

*  Evaluate X.
      S0 = XP(1)
      S1 = XP(2)
      S2 = XP(3)
      S3 = XP(4)
      S4 = XP(5)
      S5 = XP(6)
      DO I = NX0,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KX0(J,I))*FA(J)
         END DO
         S0 = S0 + ( SX0(1,I)*SIN(A) + SX0(2,I)*COS(A) )
      END DO
      DO I = NX1,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KX1(J,I))*FA(J)
         END DO
         S1 = S1 + ( SX1(1,I)*SIN(A) + SX1(2,I)*COS(A) )
      END DO
      DO I = NX2,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KX2(J,I))*FA(J)
         END DO
         S2 = S2 + ( SX2(1,I)*SIN(A) + SX2(2,I)*COS(A) )
      END DO
      DO I = NX3,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KX3(J,I))*FA(J)
         END DO
         S3 = S3 + ( SX3(1,I)*SIN(A) + SX3(2,I)*COS(A) )
      END DO
      DO I = NX4,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KX4(J,I))*FA(J)
         END DO
         S4 = S4 + ( SX4(1,I)*SIN(A) + SX4(2,I)*COS(A) )
      END DO
      X = ( S0 +
     :    ( S1 +
     :    ( S2 +
     :    ( S3 +
     :    ( S4 +
     :      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R

*  Evaluate Y.
      S0 = YP(1)
      S1 = YP(2)
      S2 = YP(3)
      S3 = YP(4)
      S4 = YP(5)
      S5 = YP(6)
      DO I = NY0,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KY0(J,I))*FA(J)
         END DO
         S0 = S0 + ( SY0(1,I)*SIN(A) + SY0(2,I)*COS(A) )
      END DO
      DO I = NY1,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KY1(J,I))*FA(J)
         END DO
         S1 = S1 + ( SY1(1,I)*SIN(A) + SY1(2,I)*COS(A) )
      END DO
      DO I = NY2,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KY2(J,I))*FA(J)
         END DO
         S2 = S2 + ( SY2(1,I)*SIN(A) + SY2(2,I)*COS(A) )
      END DO
      DO I = NY3,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KY3(J,I))*FA(J)
         END DO
         S3 = S3 + ( SY3(1,I)*SIN(A) + SY3(2,I)*COS(A) )
      END DO
      DO I = NY4,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KY4(J,I))*FA(J)
         END DO
         S4 = S4 + ( SY4(1,I)*SIN(A) + SY4(2,I)*COS(A) )
      END DO
      Y = ( S0 +
     :    ( S1 +
     :    ( S2 +
     :    ( S3 +
     :    ( S4 +
     :      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R

*  Evaluate S.
      S0 = SP(1)
      S1 = SP(2)
      S2 = SP(3)
      S3 = SP(4)
      S4 = SP(5)
      S5 = SP(6)

      DO I = NS0,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KS0(J,I))*FA(J)
         END DO
         S0 = S0 + ( SS0(1,I)*SIN(A) + SS0(2,I)*COS(A) )
      END DO
      DO I = NS1,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KS1(J,I))*FA(J)
         END DO
         S1 = S1 + ( SS1(1,I)*SIN(A) + SS1(2,I)*COS(A) )
      END DO
      DO I = NS2,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KS2(J,I))*FA(J)
         END DO
         S2 = S2 + ( SS2(1,I)*SIN(A) + SS2(2,I)*COS(A) )
      END DO
      DO I = NS3,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KS3(J,I))*FA(J)
         END DO
         S3 = S3 + ( SS3(1,I)*SIN(A) + SS3(2,I)*COS(A) )
      END DO
      DO I = NS4,1,-1
         A = 0D0
         DO J=1,14
            A = A + DBLE(KS4(J,I))*FA(J)
         END DO
         S4 = S4 + ( SS4(1,I)*SIN(A) + SS4(2,I)*COS(A) )
      END DO
      S = ( S0 +
     :    ( S1 +
     :    ( S2 +
     :    ( S3 +
     :    ( S4 +
     :      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R - X*Y/2D0

      END